/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.utils;

import java.io.Serializable;

/**
 * @author Tuukka Lehtonen
 */
public class TextSegment implements Serializable {

    private static final long                 serialVersionUID = 1428883453597424952L;

    public static transient final TextSegment EMPTY            = new TextSegment(0, 0);

    private final int start;
    private final int length;

    public static TextSegment whole(String s) {
        return new TextSegment(0, s.length());
    }

    public static TextSegment single(int pos) {
        return new TextSegment(pos, 1);
    }

    public static TextSegment before(int pos) {
        return new TextSegment(0, pos);
    }

    public TextSegment(int start, int length) {
        this.start = start;
        this.length = length;
    }

    public int start() {
        return start;
    }

    public int end() {
        return start + length;
    }

    public int length() {
        return length;
    }

    public boolean isEmpty() {
        return length == 0;
    }

    public boolean isWhole(String s) {
        if (s == null)
            return false;
        return start == 0 && length == s.length();
    }

    public boolean atStart(int pos) {
        return start == pos;
    }

    public boolean atEnd(int pos) {
        return end() == pos;
    }

    public boolean existsIn(String s) {
        int l = s != null ? s.length() : 0;
        if (start > l)
            return false;
        int end = start + length;
        if (end > l)
            return false;
        return true;
    }

    public String removeFrom(String s) {
        if (!existsIn(s))
            throw new IllegalArgumentException("segment " + this + " is out of bounds in string '" + s + "'");
        if (length == 0)
            return s;
        if (start == 0)
            return s.substring(length);
        return s.substring(0, start) + s.substring(start + length);
    }

    public String before(String s) {
        if (start == 0)
            return "";
        return s.substring(0, start);
    }

    public String after(String s) {
        if (end() == s.length())
            return "";
        return s.substring(start, start + length);
    }

    public String of(String s) {
        return s.substring(start, start + length);
    }

    public TextSegment beforeStart() {
        if (start == 0)
            return EMPTY;
        return new TextSegment(0, start);
    }

    public TextSegment afterEnd(String s) {
        int l = s.length();
        int end = end();
        if (end > l)
            throw new IllegalArgumentException("end " + end + " is beyond string '" + s + "'");
        if (end == l)
            return EMPTY;
        return new TextSegment(end, l - end);
    }

    public TextSegment prepend(int count) {
        if (count == 0)
            return this;
        int newStart = start - count;
        if (newStart < 0)
            throw new IllegalArgumentException("prepending " + this + " by " + count + " procudes invalid segment start " + newStart);
        int newLength = length + count;
        if (newLength < 0)
            throw new IllegalArgumentException("prepending " + this + " by " + count + " produces negative length: " + newLength);
        return new TextSegment(start - count, newLength);
    }

    public TextSegment append(int count) {
        if (count == 0)
            return this;
        int newLength = length + count;
        if (newLength < 0)
            throw new IllegalArgumentException("appending by " + count + " produces negative length: " + newLength);
        return new TextSegment(start, newLength);
    }

    public TextSegment extendToEnd(String s) {
        int l = s.length();
        int end = end();
        if (end > l)
            throw new IllegalArgumentException("end " + end + " is beyond string '" + s + "'");
        if (end == l)
            return this;
        return new TextSegment(start, l - start);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + length;
        result = prime * result + start;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        TextSegment other = (TextSegment) obj;
        if (length != other.length)
            return false;
        if (start != other.start)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "[" + start + "," + (start + length) + "]"; 
    }

}
