package org.simantics.document.swt.core.widget;

import java.util.Collection;

import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.simantics.datatypes.literal.Font;
import org.simantics.datatypes.literal.RGB;
import org.simantics.document.server.JSONObject;
import org.simantics.document.server.handler.AbstractEventHandler;
import org.simantics.document.server.io.CommandContextImpl;
import org.simantics.document.swt.core.SWTDocument;
import org.simantics.document.swt.core.base.LeafWidgetManager;
import org.simantics.utils.strings.StringInputProblem;
import org.simantics.utils.strings.StringInputValidator;
import org.simantics.utils.ui.widgets.ITrackedColorProvider;
import org.simantics.utils.ui.widgets.TrackedModifyEvent;
import org.simantics.utils.ui.widgets.TrackedModifyListener;
import org.simantics.utils.ui.widgets.TrackedText;

public class TrackedTextWidget extends LeafWidgetManager<Text> {

	@Override
	protected void doUpdateProperties(final SWTDocument document, Text control, JSONObject object) {

		if(control.isDisposed()) return;
		
		TrackedText tt = (TrackedText)control.getData(TrackedText.TRACKED_TEXT_KEY);
		
		String text = object.getJSONField("text");

		final AbstractEventHandler onModify = object.getJSONField("onModify");
		final StringInputValidator validator = object.getJSONField("validator");
		
		RGB.Integer invalidBackground = object.getBeanJSONFieldDefault("invalidBackground", RGB.Integer.BINDING, new RGB.Integer(255,255,255));
		RGB.Integer inactiveBackground = object.getBeanJSONFieldDefault("inactiveBackground", RGB.Integer.BINDING, new RGB.Integer(255,255,255));
		RGB.Integer hoverBackground = object.getBeanJSONFieldDefault("hoverBackground", RGB.Integer.BINDING, new RGB.Integer(255,255,255));
		RGB.Integer editingBackground = object.getBeanJSONFieldDefault("editingBackground", RGB.Integer.BINDING, new RGB.Integer(255,255,255));

		RGB.Integer foreground = object.getBeanJSONFieldDefault("foreground", RGB.Integer.BINDING, new RGB.Integer(0,0,0));
		
		Font font = object.getBeanJSONFieldDefault("font", Font.BINDING, new Font("Arial",10,"Normal"));
		
		final Color invalid = document.getColor(invalidBackground);
		final Color inactive = document.getColor(inactiveBackground);
		final Color hover = document.getColor(hoverBackground);
		final Color editing = document.getColor(editingBackground);
		
		tt.setColorProvider(new ITrackedColorProvider() {
			
			@Override
			public Color getInvalidBackground() {
				return invalid;
			}
			
			@Override
			public Color getInactiveBackground() {
				return inactive;
			}
			
			@Override
			public Color getHoverBackground() {
				return hover;
			}
			
			@Override
			public Color getEditingBackground() {
				return editing;
			}
			
		});
		
		if(text != null)
		    tt.setTextWithoutNotify(text);
		else
		    System.err.println(this + " text == null");
		// This applies coloring
		tt.setEditable(true);
		
		tt.addModifyListener(new TrackedModifyListener() {
			
			@Override
			public void modifyText(TrackedModifyEvent e) {
                CommandContextImpl parameters = new CommandContextImpl();
                parameters.putString("text", e.getText());
				document.post(onModify, parameters);
			}
			
		});
		
		if(validator != null) {
			tt.setInputValidator(new IInputValidator() {

				@Override
				public String isValid(String newText) {
					Collection<StringInputProblem> problems = validator.validate(newText);
					if(problems.isEmpty()) return null;
					else if (problems.size() == 1) {
						return problems.iterator().next().getDescription();
					} else return problems.toString();
				}

			});
		}
		
		control.setFont(document.getFont(font));
		control.setForeground(document.getColor(foreground));
		
	}

	@Override
	protected Text doCreateControl(SWTDocument document, Composite parent, JSONObject object) {
        int style = SWT.NONE;

        Boolean multiLine = object.getJSONFieldDefault("multiLine", false);
        if(Boolean.TRUE.equals(multiLine))
            style |= SWT.MULTI;
        
        Boolean vscroll = object.getJSONFieldDefault("VScroll", false);
        if(Boolean.TRUE.equals(vscroll))
            style |= SWT.V_SCROLL;
        
        Boolean hscroll = object.getJSONFieldDefault("HScroll", false);
        if(Boolean.TRUE.equals(hscroll))
            style |= SWT.H_SCROLL;
        
		TrackedText tt = new TrackedText(parent, style);

		return tt.getWidget();
	}
	
}
