/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package fi.vtt.simantics.procore.internal;

import org.simantics.db.SessionReference;
import org.simantics.db.Database.Session;
import org.simantics.db.Database.Session.Information;
import org.simantics.db.Database.Session.Transaction;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;

import fi.vtt.simantics.procore.DebugPolicy;

final class GraphSessionSocket extends GraphSession {
    
    public GraphSessionSocket(SessionImplSocket sessionImpl, SessionReference sessionReference, Session dbSession)
    throws DatabaseException {
        super(sessionImpl, sessionReference, dbSession);
    }
    @Override
    protected ServerInformationImpl getServerInformation()
    throws DatabaseException {
        Information t = dbSession.getInformation();
        return new ServerInformationImpl(t.getServerId(), t.getProtocolId(), t.getDatabaseId(), t.getFirstChangeSetId());
    }
    @Override
    public long askReadTransaction(int thread)
    throws DatabaseException {
        Transaction t = dbSession.askReadTransaction();
        updateLastChangeSetId(thread, t.getHeadChangeSetId(), true);
        return t.getTransactionId();
    }
    @Override
    public long askWriteTransaction(int thread, long transactionId)
    throws DatabaseException {
        Transaction t = dbSession.askWriteTransaction(transactionId);
        updateLastChangeSetId(thread, t.getHeadChangeSetId(), true);
        return t.getTransactionId();
    }
    @Override
    public void endTransaction(long transactionId, boolean write)
    throws DatabaseException {
        long headChangeSetId = dbSession.endTransaction(transactionId);
        updateLastChangeSetId(Integer.MIN_VALUE, headChangeSetId, false);
    }
    @Override
    public void cancelCommit(long transactionId, long csid, byte[] metadata, SynchronizeContextI context)
    throws DatabaseException {
        try {
            assert(null == synchronizeContext);
            synchronizeContext = context;
            long headChangeSetId = dbSession.cancelCommit(transactionId, csid, metadata, context);
            if (csid+1 != headChangeSetId)
                Logger.defaultLogError("Client and server out of synchronisation. Client cs="
                    + (csid+1) + "server cs=" + headChangeSetId);
            metadataCache.addNext(headChangeSetId, metadata);
        } finally {
            synchronizeContext = null;
        }
    }
    @Override
    public void acceptCommit(long transactionId, long csid, byte[] metadata)
    throws DatabaseException {
        dbSession.acceptCommit(transactionId, csid, metadata);
        metadataCache.addNext(csid+1, metadata);
    }
    @Override
    public void stop()
    throws DatabaseException {
        dbSession.close();
    }
    @Override
    public long reserveIds(int count)
    throws DatabaseException {
        Logger.defaultLogTrace("Asking for ids " + count + ".");
        long firstId = dbSession.reserveIds(count);
        Logger.defaultLogTrace("First id is " + firstId + ".");
        if (DebugPolicy.REPORT_CLUSTER_ID_ALLOCATION)
            System.out.println("Client reserves new ids [" + firstId + " - " + (firstId+count-1) + "] from server.");
        return firstId;
    }
}
