/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.wiki.ui;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;


public class SimanticsDialect implements MarkupTransform {
    
    final static public SimanticsDialect INSTANCE = new SimanticsDialect();

    class TrendTransform implements MarkupTransform {

        final Pattern startPattern = Pattern.compile("\\{trend(?::([^\\}]+))?\\}");
        
        @Override
        public String apply(ReadGraph graph, Variable variable, String markup) {
            
            StringBuffer sb = new StringBuffer();
            Matcher matcher = startPattern.matcher(markup);
            while(matcher.find()) {
                matcher.appendReplacement(sb, "");
                String options = matcher.group(1);
                //String path = "";
                Map<String, String> params = new HashMap<String, String>();
                if (options != null) {
                    String[] optionPairs = options.split("\\s*\\|\\s*");
                    for (String optionPair: optionPairs) {
                        String[] keyValue = optionPair.split("\\s*=\\s*");
                        if (keyValue.length == 2) {
                            String key = keyValue[0].trim();
                            String value = keyValue[1].trim();
                            params.put(key, value);
                            /*                           
                            if (key.equals("path")) {
                                path = value;
                            }
                            */
                        }
                    }
                }
//                sb.append("[[Image:" + path + "]]"); // FIXME: fetch value from somewhere...
                int width = 300;
                int height = 200;
                String title="Trend";
                if(params.containsKey("width")) width = Integer.parseInt(params.get("width"));
                if(params.containsKey("height")) height = Integer.parseInt(params.get("height"));
                if(params.containsKey("title")) title = params.get("title");
                sb.append("<object classid=\"org.simantics.wiki.ui.objects.TrendRenderer\" title=\""+title+"\" data=\""+params.get("data")+"\" width=\""+width+"\" height=\""+height+"\"/>");
             
            }
            matcher.appendTail(sb);
            return sb.toString();
            
        }
        
    }

    public static class TimestampTransform implements MarkupTransform {

        final Pattern startPattern = Pattern.compile("\\{timestamp(?::([^\\}]+))?\\}");
        
        @Override
        public String apply(ReadGraph graph, Variable variable, String markup) {
            
            StringBuffer sb = new StringBuffer();
            Matcher matcher = startPattern.matcher(markup);
            while(matcher.find()) {
                matcher.appendReplacement(sb, "");
                String options = matcher.group(1);
                DateFormat df = null;
                if (options != null) {
                	df = new SimpleDateFormat(options);
                } else {
                	df = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss Z");
                }
                sb.append(df.format(new Date()));
            }
            matcher.appendTail(sb);
            return sb.toString();
        }
        
    }

    
    /**
     * 
     * Syntax example: {diagram:model=Model 1|layer=Default}
     * 
     * @author J-P
     *
     */
    class DiagramTransform implements MarkupTransform {

        final Pattern startPattern = Pattern.compile("\\{diagram(?::([^\\}]+))?\\}");
        
        @Override
        public String apply(ReadGraph graph, Variable variable, String markup) {
            
            StringBuffer sb = new StringBuffer();
            Matcher matcher = startPattern.matcher(markup);
            while(matcher.find()) {
                matcher.appendReplacement(sb, "");
                String options = matcher.group(1);
                //String path = "";
                Map<String, String> params = new HashMap<String, String>();
                if (options != null) {
                    String[] optionPairs = options.split("\\s*\\|\\s*");
                    for (String optionPair: optionPairs) {
                        String[] keyValue = optionPair.split("\\s*=\\s*");
                        if (keyValue.length == 2) {
                            String key = keyValue[0].trim();
                            String value = keyValue[1].trim();
                            params.put(key, value);
                            /*                           
                            if (key.equals("path")) {
                                path = value;
                            }
                            */
                        }
                    }
                }
//                sb.append("[[Image:" + path + "]]"); // FIXME: fetch value from somewhere...
                String data = "resource="+params.get("resource");
                if(params.containsKey("layer"))
                	data += "|layer="+params.get("layer");
                sb.append("<p style=\"height:768\"><object classid=\"org.simantics.wiki.ui.objects.DiagramRenderer\" data=\""+data+"\" width=\"1024\" height=\"768\"/></p>");
            }
            matcher.appendTail(sb);
            return sb.toString();
            
        }
        
    }
    
    /**
     * 
     * Syntax example: {variable:#HasName}
     * 
     * @author Antti
     *
     */
    class VariableTransform implements MarkupTransform {

        final Pattern startPattern = Pattern.compile("\\{variable(?::([^\\}]+))?\\}");
        
        @Override
        public String apply(ReadGraph graph, Variable variable, String markup) throws DatabaseException {
            
            StringBuffer sb = new StringBuffer();
            Matcher matcher = startPattern.matcher(markup);
            while(matcher.find()) {
                matcher.appendReplacement(sb, "");
                String rvi = matcher.group(1);
                Object value = variable.browse(graph, rvi).getValue(graph);
                sb.append(value.toString());
            }
            matcher.appendTail(sb);
            return sb.toString();
            
        }
        
    }
    
    /**
     * 
     * Syntax example: {include:http://company.org/Type}
     * 
     * @author Antti
     *
     */
    class IncludeTransform implements MarkupTransform {

        final Pattern startPattern = Pattern.compile("\\{include(?::([^\\}]+))?\\}");
        
        @Override
        public String apply(ReadGraph graph, Variable variable, String markup) throws DatabaseException {
            
            StringBuffer sb = new StringBuffer();
            Matcher matcher = startPattern.matcher(markup);
            while(matcher.find()) {
                matcher.appendReplacement(sb, "");
                String uri = matcher.group(1);
                Layer0 L0 = Layer0.getInstance(graph);
                Resource resource = graph.syncRequest(new org.simantics.db.common.primitiverequest.Resource(uri));
                String desc = graph.getPossibleRelatedValue(resource, L0.HasDescription);
                sb.append(desc);
            }
            matcher.appendTail(sb);
            return sb.toString();
            
        }
        
    }

    final private MarkupTransform[] transforms = { new TrendTransform(), new DiagramTransform(), new TimestampTransform(), new VariableTransform(), new IncludeTransform() };

    @Override
    public String apply(ReadGraph graph, Variable variable, String markup) throws DatabaseException {
        for(MarkupTransform t : transforms) markup = t.apply(graph, variable, markup);
        return markup;
    }
    
}