package org.simantics.scl.compiler.internal.parsing.parser;

import java.io.DataInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import org.simantics.scl.compiler.internal.parsing.exceptions.SCLSyntaxErrorException;

import org.simantics.scl.compiler.internal.parsing.Token;

public abstract class SCLParser {    
    public static final boolean TRACE = false;

    private static final int INITIAL_CAPACITY = 16;
    private static final int STATE_COUNT = 362;
    private static final int TERMINAL_COUNT = 85;
    private static final int NONTERMINAL_COUNT = 52;
    private static final int PRODUCT_COUNT = 138;
    
    private static final int[] ACTION_ROW_ID = new int[STATE_COUNT];
    private static final int[] ACTION_COLUMN_ID = new int[TERMINAL_COUNT];
    private static final short[] ACTION_TABLE = new short[6832];
    private static final int[] ERROR_TABLE = new int[962];
    private static final int[] GOTO_ROW_ID = new int[STATE_COUNT];
    private static final int[] GOTO_COLUMN_ID = new int[NONTERMINAL_COUNT];
    private static final short[] GOTO_TABLE = new short[1953];
    private static final int[] PRODUCT_LHS = new int[PRODUCT_COUNT];

    private static final short STATE_MASK = (short)0x0fff;
    private static final short REDUCE_MASK = (short)0x8000;
    private static final short POP_MASK = (short)0x4000;
    private static final short PUSH_MASK = (short)0x2000;
    private static final short ERROR_ACTION = (short)0xffff;
    private static final short ACCEPT_ACTION = (short)0xfffe;
    
    public static final String[] TERMINAL_NAMES = new String[] {
        "SEMICOLON",
        "LBRACE",
        "RBRACE",
        "MODULE",
        "COMMA",
        "HASTYPE",
        "DATA",
        "ID",
        "EQUALS",
        "BAR",
        "TYPE",
        "CLASS",
        "WHERE",
        "INSTANCE",
        "DERIVING",
        "BEGIN_STRING",
        "END_STRING",
        "ANNOTATION_ID",
        "INFIX",
        "INFIXL",
        "INFIXR",
        "INTEGER",
        "IMPORTJAVA",
        "EFFECT",
        "RULE",
        "ABSTRACT_RULE",
        "EXTENDS",
        "MAPPING_RELATION",
        "FOLLOWS",
        "RULESET",
        "IMPORT",
        "INCLUDE",
        "AS",
        "LPAREN",
        "RPAREN",
        "HIDING",
        "ARROW",
        "COLON",
        "MINUS",
        "SYMBOL",
        "LESS",
        "GREATER",
        "SEPARATED_DOT",
        "ESCAPED_ID",
        "LAMBDA",
        "LAMBDA_MATCH",
        "LET",
        "IF",
        "MATCH",
        "DO",
        "MDO",
        "ENFORCE",
        "BLANK",
        "FLOAT",
        "LBRACKET",
        "ESCAPED_SYMBOL",
        "CHAR",
        "WHEN",
        "ATTACHED_HASH",
        "SELECT",
        "SELECT_FIRST",
        "SELECT_DISTINCT",
        "TRANSFORMATION",
        "EQ",
        "CHR_SELECT",
        "ATTACHED_DOT",
        "IN",
        "THEN",
        "ELSE",
        "WITH",
        "RBRACKET",
        "DOTDOT",
        "AT",
        "SUSPEND_STRING",
        "CONTINUE_STRING",
        "BINDS",
        "IMPLIES",
        "THEN_AFTER_WHEN",
        "CONSTRAINT",
        "BY",
        "QUERY_OP",
        "FORALL",
        "COMMENT",
        "EOL",
        "EOF"
    };

    public static final String[] NONTERMINAL_NAMES = new String[] {
        "module",
        "commands",
        "import",
        "type",
        "exp",
        "equationBlock",
        "declaration",
        "command",
        "statement",
        "declarations",
        "field",
        "var",
        "bexp",
        "rhs",
        "constructor",
        "context",
        "fundeps",
        "atype",
        "aexp",
        "ruleDeclarations",
        "statements",
        "importSpec",
        "importItem",
        "fieldDeclaration",
        "guardedExpEq",
        "fundep",
        "ruleDeclaration",
        "query",
        "lexp",
        "symbol",
        "faexp",
        "accessor",
        "case",
        "queryBlock",
        "verboseChrQuery",
        "stringLiteral",
        "symbolWithoutMinus",
        "listQualifier",
        "chrQuery",
        "chrQueryPart",
        "caseRhs",
        "guardedExpArrow",
        "equation",
        "etype",
        "btype",
        "dummy",
        "init$6",
        "init$5",
        "init$4",
        "init$3",
        "init$2",
        "init$1"
    };
        
    static {
        try {
            DataInputStream input = new DataInputStream(SCLParser.class.getResourceAsStream("SCLParser.dat"));
            for(int i=0;i<ACTION_ROW_ID.length;++i)
                ACTION_ROW_ID[i] = input.readInt();
            for(int i=0;i<ACTION_COLUMN_ID.length;++i)
                ACTION_COLUMN_ID[i] = input.readInt();    
            for(int i=0;i<ACTION_TABLE.length;++i)
                ACTION_TABLE[i] = input.readShort();
            for(int i=0;i<ERROR_TABLE.length;++i)
                ERROR_TABLE[i] = input.readInt();
            for(int i=0;i<GOTO_ROW_ID.length;++i)
                GOTO_ROW_ID[i] = input.readInt();
            for(int i=0;i<GOTO_COLUMN_ID.length;++i)
                GOTO_COLUMN_ID[i] = input.readInt();    
            for(int i=0;i<GOTO_TABLE.length;++i)
                GOTO_TABLE[i] = input.readShort();
            for(int i=0;i<PRODUCT_LHS.length;++i)
                PRODUCT_LHS[i] = input.readInt();
            input.close();
        } catch(IOException e) {
            e.printStackTrace();
        }
    }
    
    private static short getAction(int state, int symbol) {
        int id = TERMINAL_COUNT*state + symbol;
        if( ((ERROR_TABLE[id>>5] >> (id&31))&1) != 0 )
            return ERROR_ACTION;
        return ACTION_TABLE[ACTION_ROW_ID[state] + ACTION_COLUMN_ID[symbol]];
    }
    
    private static short getGoto(int state, int symbol) {
        return GOTO_TABLE[GOTO_ROW_ID[state] + GOTO_COLUMN_ID[symbol]];
    }
    
    protected abstract Token nextToken();
    
    private Object[] symbolStack = new Object[INITIAL_CAPACITY];
    private int symbolStackLength = 0;
    
    private int[] stateStack = new int[INITIAL_CAPACITY];
    private int[] symbolStackPositionStack = new int[INITIAL_CAPACITY];
    private int stateStackLength = 0;
    
    // For reduce
    private int reductionLength;
    
    protected int length() {
        return reductionLength;
    }
    
    protected Object get(int i) {
        if(i < 0 || i >= reductionLength)
            throw new IndexOutOfBoundsException();
        return symbolStack[symbolStackLength+i];
    }
    
    private String parseErrorDescription(int state, Token token, int tokenId) {
        StringBuilder b = new StringBuilder();
        b.append("Unexpected token '").append(token)
         .append("' (").append(TERMINAL_NAMES[tokenId])
         .append("). Expected one of ");
        ArrayList<String> possibleTerminals = new ArrayList<String>();
        for(int i=0;i<TERMINAL_COUNT;++i)
            if(getAction(state, i) != ERROR_ACTION)
                possibleTerminals.add(TERMINAL_NAMES[i]);
        Collections.sort(possibleTerminals);
        for(int i=0;i<possibleTerminals.size();++i) {
            if(i > 0)
                b.append(", ");
            b.append(possibleTerminals.get(i));
        }
        b.append('.');
        return b.toString();
    }
    
    protected abstract RuntimeException syntaxError(Token token, String description);
    
    private static String describeAction(boolean isGoto, int action) {
        if(action == ERROR_ACTION)
            return "ERROR";
        if(action == ACCEPT_ACTION)
            return "ACCEPT";
        StringBuilder b = new StringBuilder();
        if(isGoto)
            b.append("GOTO ");
        else {
            if((action & REDUCE_MASK) != 0) {
                action ^= REDUCE_MASK;
                b.append("REDUCE");
            }
            else
                b.append("SHIFT");
        }
        if((action & POP_MASK) != 0) {
            action ^= POP_MASK;
            b.append(" POP");
        }
        if((action & PUSH_MASK) != 0) {
            action ^= PUSH_MASK;
            b.append(" PUSH");
        }
        b.append(' ').append(action);
        return b.toString();
    }
    
    private void printState(int state) {
        System.out.print("state=" + state + ":");
        for(int i=symbolStackLength-1,j=stateStackLength-1;i>=0;--i) {
            Object s = symbolStack[i];
            if(s instanceof Token)
                System.out.print(" " + TERMINAL_NAMES[((Token)s).id]);
            else if(s == null)
                System.out.print(" null");
            else
                System.out.print(" " + s.getClass().getSimpleName());
            while(j>=0 && symbolStackPositionStack[j]==i)
                System.out.print(" (" + stateStack[j--] + ")");
        }
        System.out.println();
    }
    
    private Object parse(int state) {
        while(true) {
            Token token = nextToken();
            int tokenId = token.id;
            if(TRACE)
                System.out.println("---> token " + TERMINAL_NAMES[tokenId] + " \"" + token.text + "\" <---");
            while(true) {
                if(TRACE)
                    printState(state);
                short action = getAction(state, tokenId);
                if(TRACE)
                    System.out.println("    -> action=" + describeAction(false, action));
                //System.out.println(STATE_DESCRIPTIONS[state]);
                if((action & REDUCE_MASK) != 0) {
                    if(action == ACCEPT_ACTION)
                        return symbolStack[symbolStackLength-1];
                    if(action == ERROR_ACTION)
                        throw syntaxError(token, parseErrorDescription(state, token, tokenId));
                    int popAmount = (action >>> 13)&3;
                    if(TRACE) {
                        if(popAmount > 0)
                            System.out.println("    POP " + popAmount);
                    }
                    stateStackLength -= popAmount;
                    action &= STATE_MASK;
                    
                    int reductionBegin = symbolStackPositionStack[--stateStackLength];
                    
                    reductionLength = symbolStackLength-reductionBegin;
                    symbolStackLength = reductionBegin;
                    
                    if(symbolStackLength == symbolStack.length)
                        symbolStack = Arrays.copyOf(symbolStack, symbolStackLength*2);
                    Object symbol = reduce(action);
                    postReduce(symbol);
                    symbolStack[symbolStackLength] = symbol;
                    
                    state = stateStack[stateStackLength];
                    if(TRACE) {
                        ++symbolStackLength;
                        printState(state);
                        --symbolStackLength;
                        System.out.println("    nonterminal=" + NONTERMINAL_NAMES[PRODUCT_LHS[action]]);
                    }
                    action = getGoto(state, PRODUCT_LHS[action]);
                    if(TRACE)
                        System.out.println("    -> action=" + describeAction(true, action));
                        
                    // Pop state
                    if((action & POP_MASK) != 0) {
                        --stateStackLength;
                    }
                    // Push state
                    if((action & PUSH_MASK) != 0) {
                        if(stateStackLength == stateStack.length) {
                            stateStack = Arrays.copyOf(stateStack, stateStackLength*2);
                            symbolStackPositionStack = Arrays.copyOf(symbolStackPositionStack, stateStackLength*2);
                        }
                        symbolStackPositionStack[stateStackLength] = symbolStackLength;
                        stateStack[stateStackLength++] = state;
                    }
                    state = action & STATE_MASK;
                    ++symbolStackLength;
                }
                else {
                    // Pop state
                    if((action & POP_MASK) != 0) {
                        --stateStackLength;
                    }
                    // Push state
                    if((action & PUSH_MASK) != 0) {
                        if(stateStackLength == stateStack.length) {
                            stateStack = Arrays.copyOf(stateStack, stateStackLength*2);
                            symbolStackPositionStack = Arrays.copyOf(symbolStackPositionStack, stateStackLength*2);
                        }
                        symbolStackPositionStack[stateStackLength] = symbolStackLength;
                        stateStack[stateStackLength++] = state;
                    }
                    
                    // New state
                    state = action & STATE_MASK;
                    
                    // Push symbol
                    if(symbolStackLength == symbolStack.length)
                        symbolStack = Arrays.copyOf(symbolStack, symbolStackLength*2);
                    symbolStack[symbolStackLength++] = token;
                    break;
                }
            }
        }
    }
    
    public Object parseModule() {
        return parse(0);
    }
    public Object parseCommands() {
        return parse(346);
    }
    public Object parseImport() {
        return parse(354);
    }
    public Object parseType() {
        return parse(356);
    }
    public Object parseExp() {
        return parse(358);
    }
    public Object parseEquationBlock() {
        return parse(360);
    }


    protected Object reduce(int productionId) {
        try {
        switch(productionId) {
        case 0:
            return reduceModule();
        case 1:
            return reduceOneCommand();
        case 2:
            return reduceManyCommands();
        case 3:
            return reduceImport();
        case 4:
            return reduceArrow();
        case 5:
            return reduceLocalTypeAnnotation();
        case 6:
            return reduceEquationBlock();
        case 7:
            return reduceModuleHeader();
        case 8:
            return reduceTypeAnnotation();
        case 9:
            return reduceValueDefinition();
        case 10:
            return reduceDataDefinition();
        case 11:
            return reduceTypeDefinition();
        case 12:
            return reduceClassDefinition();
        case 13:
            return reduceInstanceDefinition();
        case 14:
            return reduceDerivingInstanceDefinition();
        case 15:
            return reduceDocumentationString();
        case 16:
            return reduceAnnotation();
        case 17:
            return reducePrecedenceDefinition();
        case 18:
            return reduceJustImport();
        case 19:
            return reduceImportJava();
        case 20:
            return reduceEffectDefinition();
        case 21:
            return reduceRuleDefinition();
        case 22:
            return reduceMappingRelationDefinition();
        case 23:
            return reduceRelationDefinition();
        case 24:
            return reduceRulesetDefinition();
        case 25:
            return reduceStatementCommand();
        case 26:
            return reduceImportCommand();
        case 27:
            return reduceGuardStatement();
        case 28:
            return reduceLetStatement();
        case 29:
            return reduceBindStatement();
        case 30:
            return reduceRuleStatement();
        case 31:
            return reduceCHRStatement();
        case 32:
            return reduceVerboseCHRStatement();
        case 33:
            return reduceConstraintStatement();
        case 34:
            return reduceLocalInclude();
        case 35:
            return reduceDeclarations();
        case 36:
            return reduceField();
        case 37:
            return reduceFieldShorthand();
        case 38:
            return reduceVarId();
        case 39:
            return reduceEscapedSymbol();
        case 40:
            return reduceTupleConstructor();
        case 41:
            return reduceBinary();
        case 42:
            return reduceSimpleRhs();
        case 43:
            return reduceGuardedRhs();
        case 44:
            return reduceConstructor();
        case 45:
            return reduceRecordConstructor();
        case 46:
            return reduceContext();
        case 47:
            return reduceFundeps();
        case 48:
            return reduceTypeVar();
        case 49:
            return reduceTupleType();
        case 50:
            return reduceListType();
        case 51:
            return reduceListTypeConstructor();
        case 52:
            return reduceTupleTypeConstructor();
        case 53:
            return reduceLambda();
        case 54:
            return reduceLambdaMatch();
        case 55:
            return reduceLet();
        case 56:
            return reduceIf();
        case 57:
            return reduceMatch();
        case 58:
            return reduceDo();
        case 59:
            return reduceSelect();
        case 60:
            return reduceCHRSelect();
        case 61:
            return reduceEnforce();
        case 62:
            return reduceVar();
        case 63:
            return reduceHashedId();
        case 64:
            return reduceBlank();
        case 65:
            return reduceInteger();
        case 66:
            return reduceFloat();
        case 67:
            return reduceString();
        case 68:
            return reduceChar();
        case 69:
            return reduceTuple();
        case 70:
            return reduceViewPattern();
        case 71:
            return reduceRightSection();
        case 72:
            return reduceLeftSection();
        case 73:
            return reduceListLiteral();
        case 74:
            return reduceRange();
        case 75:
            return reduceListComprehension();
        case 76:
            return reduceAs();
        case 77:
            return reduceRecord();
        case 78:
            return reduceTransformation();
        case 79:
            return reduceEq();
        case 80:
            return reduceRuleDeclarations();
        case 81:
            return reduceStatements();
        case 82:
            return reduceImportShowing();
        case 83:
            return reduceImportHiding();
        case 84:
            return reduceImportValueItem();
        case 85:
            return reduceFieldDescription();
        case 86:
            return reduceGuardedExpEq();
        case 87:
            return reduceFundep();
        case 88:
            return reduceQueryRuleDeclaration();
        case 89:
            return reduceAnnotation();
        case 90:
            return reduceGuardQuery();
        case 91:
            return reduceEqualsQuery();
        case 92:
            return reduceBindQuery();
        case 93:
            return reduceCompositeQuery();
        case 94:
            return reduceApply();
        case 95:
            return reduceSymbol();
        case 96:
            return reduceEscapedId();
        case 97:
            return reduceMinus();
        case 98:
            return reduceLess();
        case 99:
            return reduceGreater();
        case 100:
            return reduceDot();
        case 101:
            return reduceFieldAccess();
        case 102:
            return reduceIdAccessor();
        case 103:
            return reduceStringAccessor();
        case 104:
            return reduceExpAccessor();
        case 105:
            return reduceCase();
        case 106:
            return reduceQueryBlock();
        case 107:
            return reduceVerboseCHRConjunction();
        case 108:
            return reduceStringLiteral();
        case 109:
            return reduceSymbol();
        case 110:
            return reduceEscapedId();
        case 111:
            return reduceLess();
        case 112:
            return reduceGreater();
        case 113:
            return reduceDot();
        case 114:
            return reduceGuardQualifier();
        case 115:
            return reduceLetQualifier();
        case 116:
            return reduceBindQualifier();
        case 117:
            return reduceThenQualifier();
        case 118:
            return reduceCHRConjunction();
        case 119:
            return reduceCHRAtom();
        case 120:
            return reduceCHREquals();
        case 121:
            return reduceCHRBinds();
        case 122:
            return reduceSimpleCaseRhs();
        case 123:
            return reduceGuardedCaseRhs();
        case 124:
            return reduceGuardedExpArrow();
        case 125:
            return reduceGuardEquation();
        case 126:
            return reduceBasicEquation();
        case 127:
            return reduceEffect();
        case 128:
            return reduceJustEtype();
        case 129:
            return reduceForAll();
        case 130:
            return reduceApplyType();
        case 131:
            return reduceDummy();

        default:
            throw new RuntimeException("Internal parser error.");
        }
        } catch(SCLSyntaxErrorException e) {
            StringBuilder b = new StringBuilder();
            b.append("Failed to reduce");
            for(int i=0;i<length();++i) {
                Object obj = get(i);
                b.append("\n    (").append(i).append(") \"").append(obj).append('\"');
                if(obj instanceof Token)
                    b.append(" (").append(TERMINAL_NAMES[((Token)obj).id]).append(")");
                else
                    b.append(" [").append(obj.getClass().getSimpleName()).append("]");
            }
            throw new RuntimeException(b.toString(), e);
        } 
    }

    /**
     * module ::= (declaration (SEMICOLON declaration)&#42;)?
     */
    protected abstract Object reduceModule();
    /**
     * commands ::= command?
     */
    protected abstract Object reduceOneCommand();
    /**
     * commands ::= commands SEMICOLON command
     */
    protected abstract Object reduceManyCommands();
    /**
     * import ::= (IMPORT | INCLUDE) BEGIN_STRING END_STRING (AS ID)? importSpec?
     */
    protected abstract Object reduceImport();
    /**
     * type ::= (etype (ARROW | IMPLIES))&#42; etype
     */
    protected abstract Object reduceArrow();
    /**
     * exp ::= bexp (HASTYPE type)?
     */
    protected abstract Object reduceLocalTypeAnnotation();
    /**
     * equationBlock ::= (equation (SEMICOLON equation)&#42;)?
     */
    protected abstract Object reduceEquationBlock();
    /**
     * declaration ::= MODULE LBRACE (field (COMMA field)&#42;)? RBRACE
     */
    protected abstract Object reduceModuleHeader();
    /**
     * declaration ::= (var COMMA)&#42; var HASTYPE type
     */
    protected abstract Object reduceTypeAnnotation();
    /**
     * declaration ::= bexp rhs
     */
    protected abstract Object reduceValueDefinition();
    /**
     * declaration ::= DATA ID ID&#42; (EQUALS (constructor BAR)&#42; constructor)?
     */
    protected abstract Object reduceDataDefinition();
    /**
     * declaration ::= TYPE ID ID&#42; EQUALS type
     */
    protected abstract Object reduceTypeDefinition();
    /**
     * declaration ::= CLASS context? ID ID&#42; (BAR fundeps | (BAR fundeps)? WHERE declarations)?
     */
    protected abstract Object reduceClassDefinition();
    /**
     * declaration ::= INSTANCE context? ID atype atype&#42; (WHERE declarations)?
     */
    protected abstract Object reduceInstanceDefinition();
    /**
     * declaration ::= DERIVING INSTANCE context? ID atype atype&#42;
     */
    protected abstract Object reduceDerivingInstanceDefinition();
    /**
     * declaration ::= BEGIN_STRING END_STRING
     */
    protected abstract Object reduceDocumentationString();
    /**
     * declaration ::= ANNOTATION_ID aexp&#42;
     */
    protected abstract Object reduceAnnotation();
    /**
     * declaration ::= (INFIX | INFIXL | INFIXR) INTEGER (var COMMA)&#42; var
     */
    protected abstract Object reducePrecedenceDefinition();
    /**
     * declaration ::= import
     */
    protected abstract Object reduceJustImport();
    /**
     * declaration ::= IMPORTJAVA BEGIN_STRING END_STRING WHERE declarations
     */
    protected abstract Object reduceImportJava();
    /**
     * declaration ::= EFFECT ID BEGIN_STRING END_STRING BEGIN_STRING END_STRING
     */
    protected abstract Object reduceEffectDefinition();
    /**
     * declaration ::= (RULE | ABSTRACT_RULE) ID (EXTENDS (ID COMMA)&#42; ID)? WHERE ruleDeclarations
     */
    protected abstract Object reduceRuleDefinition();
    /**
     * declaration ::= MAPPING_RELATION ID atype&#42;
     */
    protected abstract Object reduceMappingRelationDefinition();
    /**
     * declaration ::= bexp FOLLOWS ruleDeclarations
     */
    protected abstract Object reduceRelationDefinition();
    /**
     * declaration ::= RULESET ID WHERE statements
     */
    protected abstract Object reduceRulesetDefinition();
    /**
     * command ::= statement
     */
    protected abstract Object reduceStatementCommand();
    /**
     * command ::= import
     */
    protected abstract Object reduceImportCommand();
    /**
     * statement ::= exp
     */
    protected abstract Object reduceGuardStatement();
    /**
     * statement ::= exp rhs
     */
    protected abstract Object reduceLetStatement();
    /**
     * statement ::= exp BINDS exp
     */
    protected abstract Object reduceBindStatement();
    /**
     * statement ::= exp FOLLOWS queryBlock
     */
    protected abstract Object reduceRuleStatement();
    /**
     * statement ::= chrQuery IMPLIES chrQuery
     */
    protected abstract Object reduceCHRStatement();
    /**
     * statement ::= WHEN verboseChrQuery THEN_AFTER_WHEN verboseChrQuery
     */
    protected abstract Object reduceVerboseCHRStatement();
    /**
     * statement ::= CONSTRAINT constructor
     */
    protected abstract Object reduceConstraintStatement();
    /**
     * statement ::= INCLUDE ID aexp
     */
    protected abstract Object reduceLocalInclude();
    /**
     * declarations ::= LBRACE (declaration (SEMICOLON (declaration SEMICOLON)&#42; declaration)?)? RBRACE
     */
    protected abstract Object reduceDeclarations();
    /**
     * field ::= ID EQUALS exp
     */
    protected abstract Object reduceField();
    /**
     * field ::= ID
     */
    protected abstract Object reduceFieldShorthand();
    /**
     * var ::= ID
     */
    protected abstract Object reduceVarId();
    /**
     * var ::= ESCAPED_SYMBOL
     */
    protected abstract Object reduceEscapedSymbol();
    /**
     * var ::= LPAREN COMMA COMMA&#42; RPAREN
     */
    protected abstract Object reduceTupleConstructor();
    /**
     * bexp ::= MINUS? lexp (symbol lexp)&#42;
     */
    protected abstract Object reduceBinary();
    /**
     * rhs ::= EQUALS exp (WHERE statements)?
     */
    protected abstract Object reduceSimpleRhs();
    /**
     * rhs ::= guardedExpEq guardedExpEq&#42; (WHERE statements)?
     */
    protected abstract Object reduceGuardedRhs();
    /**
     * constructor ::= (ANNOTATION_ID aexp)&#42; ID atype&#42;
     */
    protected abstract Object reduceConstructor();
    /**
     * constructor ::= (ANNOTATION_ID aexp)&#42; ID LBRACE fieldDeclaration (COMMA fieldDeclaration)&#42; RBRACE
     */
    protected abstract Object reduceRecordConstructor();
    /**
     * context ::= LPAREN type (COMMA type)&#42; RPAREN IMPLIES
     */
    protected abstract Object reduceContext();
    /**
     * fundeps ::= (fundep COMMA)&#42; fundep
     */
    protected abstract Object reduceFundeps();
    /**
     * atype ::= ID
     */
    protected abstract Object reduceTypeVar();
    /**
     * atype ::= LPAREN (type (COMMA (type COMMA)&#42; type)?)? RPAREN
     */
    protected abstract Object reduceTupleType();
    /**
     * atype ::= LBRACKET type RBRACKET
     */
    protected abstract Object reduceListType();
    /**
     * atype ::= LBRACKET RBRACKET
     */
    protected abstract Object reduceListTypeConstructor();
    /**
     * atype ::= LPAREN COMMA COMMA&#42; RPAREN
     */
    protected abstract Object reduceTupleTypeConstructor();
    /**
     * aexp ::= LAMBDA aexp aexp&#42; ARROW exp
     */
    protected abstract Object reduceLambda();
    /**
     * aexp ::= LAMBDA_MATCH LBRACE case (SEMICOLON case)&#42; RBRACE
     */
    protected abstract Object reduceLambdaMatch();
    /**
     * aexp ::= LET statements IN exp
     */
    protected abstract Object reduceLet();
    /**
     * aexp ::= IF exp THEN exp (ELSE exp)?
     */
    protected abstract Object reduceIf();
    /**
     * aexp ::= MATCH exp WITH LBRACE case (SEMICOLON case)&#42; RBRACE
     */
    protected abstract Object reduceMatch();
    /**
     * aexp ::= (DO | MDO) statements
     */
    protected abstract Object reduceDo();
    /**
     * aexp ::= (SELECT | SELECT_FIRST | SELECT_DISTINCT) exp WHERE queryBlock
     */
    protected abstract Object reduceSelect();
    /**
     * aexp ::= CHR_SELECT exp WHERE verboseChrQuery
     */
    protected abstract Object reduceCHRSelect();
    /**
     * aexp ::= ENFORCE queryBlock
     */
    protected abstract Object reduceEnforce();
    /**
     * aexp ::= var
     */
    protected abstract Object reduceVar();
    /**
     * aexp ::= ATTACHED_HASH ID
     */
    protected abstract Object reduceHashedId();
    /**
     * aexp ::= BLANK
     */
    protected abstract Object reduceBlank();
    /**
     * aexp ::= INTEGER
     */
    protected abstract Object reduceInteger();
    /**
     * aexp ::= FLOAT
     */
    protected abstract Object reduceFloat();
    /**
     * aexp ::= stringLiteral
     */
    protected abstract Object reduceString();
    /**
     * aexp ::= CHAR
     */
    protected abstract Object reduceChar();
    /**
     * aexp ::= LPAREN (exp (COMMA (exp COMMA)&#42; exp)?)? RPAREN
     */
    protected abstract Object reduceTuple();
    /**
     * aexp ::= LPAREN exp ARROW exp RPAREN
     */
    protected abstract Object reduceViewPattern();
    /**
     * aexp ::= LPAREN symbolWithoutMinus lexp RPAREN
     */
    protected abstract Object reduceRightSection();
    /**
     * aexp ::= LPAREN lexp symbol RPAREN
     */
    protected abstract Object reduceLeftSection();
    /**
     * aexp ::= LBRACKET (exp (COMMA (exp COMMA)&#42; exp)?)? RBRACKET
     */
    protected abstract Object reduceListLiteral();
    /**
     * aexp ::= LBRACKET exp DOTDOT exp RBRACKET
     */
    protected abstract Object reduceRange();
    /**
     * aexp ::= LBRACKET exp BAR listQualifier (COMMA listQualifier)&#42; RBRACKET
     */
    protected abstract Object reduceListComprehension();
    /**
     * aexp ::= ID AT aexp
     */
    protected abstract Object reduceAs();
    /**
     * aexp ::= ID LBRACE (field (COMMA field)&#42;)? RBRACE
     */
    protected abstract Object reduceRecord();
    /**
     * aexp ::= TRANSFORMATION ID WHERE queryBlock
     */
    protected abstract Object reduceTransformation();
    /**
     * aexp ::= EQ LBRACE equationBlock RBRACE
     */
    protected abstract Object reduceEq();
    /**
     * ruleDeclarations ::= LBRACE (ruleDeclaration (SEMICOLON (ruleDeclaration SEMICOLON)&#42; ruleDeclaration)?)? RBRACE
     */
    protected abstract Object reduceRuleDeclarations();
    /**
     * statements ::= LBRACE (statement (SEMICOLON (statement SEMICOLON)&#42; statement)?)? RBRACE
     */
    protected abstract Object reduceStatements();
    /**
     * importSpec ::= LPAREN (importItem (COMMA (importItem COMMA)&#42; importItem)?)? RPAREN
     */
    protected abstract Object reduceImportShowing();
    /**
     * importSpec ::= HIDING LPAREN (importItem (COMMA importItem)&#42;)? RPAREN
     */
    protected abstract Object reduceImportHiding();
    /**
     * importItem ::= ID
     */
    protected abstract Object reduceImportValueItem();
    /**
     * fieldDeclaration ::= ID HASTYPE type
     */
    protected abstract Object reduceFieldDescription();
    /**
     * guardedExpEq ::= BAR exp (COMMA exp)&#42; EQUALS exp
     */
    protected abstract Object reduceGuardedExpEq();
    /**
     * fundep ::= ID ID&#42; ARROW ID
     */
    protected abstract Object reduceFundep();
    /**
     * ruleDeclaration ::= query
     */
    protected abstract Object reduceQueryRuleDeclaration();
    /**
     * query ::= exp
     */
    protected abstract Object reduceGuardQuery();
    /**
     * query ::= exp EQUALS exp
     */
    protected abstract Object reduceEqualsQuery();
    /**
     * query ::= exp BINDS exp
     */
    protected abstract Object reduceBindQuery();
    /**
     * query ::= QUERY_OP queryBlock
     */
    protected abstract Object reduceCompositeQuery();
    /**
     * lexp ::= faexp faexp&#42;
     */
    protected abstract Object reduceApply();
    /**
     * symbol ::= SYMBOL
     */
    protected abstract Object reduceSymbol();
    /**
     * symbol ::= ESCAPED_ID
     */
    protected abstract Object reduceEscapedId();
    /**
     * symbol ::= MINUS
     */
    protected abstract Object reduceMinus();
    /**
     * symbol ::= LESS
     */
    protected abstract Object reduceLess();
    /**
     * symbol ::= GREATER
     */
    protected abstract Object reduceGreater();
    /**
     * symbol ::= SEPARATED_DOT
     */
    protected abstract Object reduceDot();
    /**
     * faexp ::= aexp ((ATTACHED_HASH | ATTACHED_DOT) accessor)&#42;
     */
    protected abstract Object reduceFieldAccess();
    /**
     * accessor ::= ID
     */
    protected abstract Object reduceIdAccessor();
    /**
     * accessor ::= BEGIN_STRING END_STRING
     */
    protected abstract Object reduceStringAccessor();
    /**
     * accessor ::= LPAREN exp RPAREN
     */
    protected abstract Object reduceExpAccessor();
    /**
     * case ::= exp caseRhs
     */
    protected abstract Object reduceCase();
    /**
     * queryBlock ::= LBRACE (query (SEMICOLON (query SEMICOLON)&#42; query)?)? RBRACE
     */
    protected abstract Object reduceQueryBlock();
    /**
     * verboseChrQuery ::= LBRACE chrQuery (SEMICOLON chrQuery)&#42; RBRACE
     */
    protected abstract Object reduceVerboseCHRConjunction();
    /**
     * stringLiteral ::= BEGIN_STRING (SUSPEND_STRING exp CONTINUE_STRING)&#42; END_STRING
     */
    protected abstract Object reduceStringLiteral();
    /**
     * listQualifier ::= exp
     */
    protected abstract Object reduceGuardQualifier();
    /**
     * listQualifier ::= exp EQUALS exp
     */
    protected abstract Object reduceLetQualifier();
    /**
     * listQualifier ::= exp BINDS exp
     */
    protected abstract Object reduceBindQualifier();
    /**
     * listQualifier ::= THEN exp (BY exp)?
     */
    protected abstract Object reduceThenQualifier();
    /**
     * chrQuery ::= (chrQueryPart COMMA)&#42; chrQueryPart
     */
    protected abstract Object reduceCHRConjunction();
    /**
     * chrQueryPart ::= exp
     */
    protected abstract Object reduceCHRAtom();
    /**
     * chrQueryPart ::= exp EQUALS exp
     */
    protected abstract Object reduceCHREquals();
    /**
     * chrQueryPart ::= exp BINDS exp
     */
    protected abstract Object reduceCHRBinds();
    /**
     * caseRhs ::= ARROW exp (WHERE statements)?
     */
    protected abstract Object reduceSimpleCaseRhs();
    /**
     * caseRhs ::= guardedExpArrow guardedExpArrow&#42; (WHERE statements)?
     */
    protected abstract Object reduceGuardedCaseRhs();
    /**
     * guardedExpArrow ::= BAR exp (COMMA exp)&#42; ARROW exp
     */
    protected abstract Object reduceGuardedExpArrow();
    /**
     * equation ::= exp
     */
    protected abstract Object reduceGuardEquation();
    /**
     * equation ::= exp EQUALS exp
     */
    protected abstract Object reduceBasicEquation();
    /**
     * etype ::= LESS ID (COMMA ID)&#42; GREATER btype
     */
    protected abstract Object reduceEffect();
    /**
     * etype ::= btype
     */
    protected abstract Object reduceJustEtype();
    /**
     * etype ::= FORALL ID ID&#42; (SEPARATED_DOT | ATTACHED_DOT) type
     */
    protected abstract Object reduceForAll();
    /**
     * btype ::= atype atype&#42;
     */
    protected abstract Object reduceApplyType();
    /**
     * dummy ::= COMMENT EOL
     */
    protected abstract Object reduceDummy();

    protected void postReduce(Object reduced) {
    }

}
