package org.simantics.scl.compiler.internal.header;

import java.util.List;

import org.simantics.scl.compiler.elaboration.expressions.EVar;
import org.simantics.scl.compiler.elaboration.expressions.annotations.AnnotationUtils;
import org.simantics.scl.compiler.elaboration.expressions.records.FieldAssignment;
import org.simantics.scl.compiler.errors.ErrorLog;

public class ModuleHeader {
    public String classLoader;
    public long classLoaderLocation;
    public String defaultLocalName;
    public List<EVar> export;
    
    // Features
    public boolean chr;
    public boolean fields;
    
    private void read(ErrorLog errorLog, FieldAssignment[] fields) {
        for(FieldAssignment assignment : fields)
            switch(assignment.name) {
            case "bundle":
                if(assignment.value == null)
                    errorLog.log(assignment.location, "Property classLoader needs to be given a string value.");
                else {
                    classLoader = AnnotationUtils.extractString(assignment.value);
                    if(classLoader == null)
                        errorLog.log(assignment.value.location, "Expected bundle name here.");
                    else 
                        classLoaderLocation = assignment.location;
                }
                break;
            case "export":
                if(assignment.value == null)
                    errorLog.log(assignment.location, "Property export needs to be given a list of exported symbols.");
                else {
                    export = AnnotationUtils.extractIdentifierList(assignment.value);
                    if(export == null)
                        errorLog.log(assignment.value.location, "Expected a list of exported symbols.");
                }
                break;
            case "features":
                if(assignment.value == null)
                    errorLog.log(assignment.location, "Property features needs to be given a list of features (identifiers).");
                else {
                    List<EVar> features = AnnotationUtils.extractIdentifierList(assignment.value);
                    if(features == null)
                        errorLog.log(assignment.value.location, "Expected a list of features (identifiers).");
                    for(EVar feature : features)
                        handleFeature(errorLog, feature);
                }
                break;
            case "defaultLocalName":
                if(assignment.value == null)
                    errorLog.log(assignment.location, "Property defaultLocalName needs to be given a string value.");
                else {
                    defaultLocalName = AnnotationUtils.extractString(assignment.value);
                    if(defaultLocalName == null)
                        errorLog.log(assignment.value.location, "Expected string here.");
                }
                break;
            default:
                errorLog.logWarning(assignment.location, "Unknown module header field was skipped.");
            }
    }
    
    private void handleFeature(ErrorLog errorLog, EVar feature) {
        switch(feature.name) {
        case "chr": chr = true; break;
        case "fields": fields = true; break;
        default:
            errorLog.log(feature.location, "Unknown feature " + feature.name + ".");
        }
    }

    public static ModuleHeader process(ErrorLog errorLog, FieldAssignment[] fields) {
        if(fields == null)
            return null;
        ModuleHeader result = new ModuleHeader();
        result.read(errorLog, fields);
        return result;
    }
}
