package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.expressions.list.CompiledQualifier;
import org.simantics.scl.compiler.elaboration.expressions.list.ListQualifier;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;

public class EListComprehension extends SimplifiableExpression {

    public Expression head;
    public ListQualifier qualifier;
    
    public EListComprehension(Expression head, ListQualifier qualifier) {
        this.head = head;
        this.qualifier = qualifier;
    }

    public EListComprehension(long loc, Expression head,
            ListQualifier qualifier) {
        super(loc);
        this.head = head;
        this.qualifier = qualifier;
    }
    
    @Override
    public Expression checkBasicType(TypingContext context, Type requiredType) {
        qualifier.checkType(context);
        Type componentType;
        try {
            componentType = Types.unifyApply(Types.LIST, requiredType);
        } catch (MatchException e) {
            context.getErrorLog().log(location, "Expected a value with type " + requiredType + " but got a list.");
            return new EError(location);
        }
        head = head.checkType(context, componentType);
        return this;
    }
    
    @Override
    protected void updateType() throws MatchException {
        setType(Types.list(head.getType()));
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        context.pushLocation(location);
        try {
            CompiledQualifier cq = qualifier.compile(context);            
            Expression exp = context.mapList(
                    context.lambda(cq.pattern, head), 
                    cq.value);
            //System.out.println("simplified: " + exp);
            /*try {
                exp.validateType(context.getEnvironment());
            } catch (TypeValidationException e) {
                throw new InternalCompilerError(e);
            }*/
            exp = exp.simplify(context);
            return exp;
        } finally {
            context.popLocation();
        }
    }

    @Override
    public Expression resolve(TranslationContext context) {
        context.pushFrame();
        qualifier.resolve(context);
        head = head.resolve(context);
        context.popFrame();
        return this;
    }

    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            head.setLocationDeep(loc);
            qualifier.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }

}
