package org.simantics.modeling.utils;

import java.util.Collection;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.CommonDBUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.adapters.ChangeHistoryUpdated;
import org.simantics.modeling.adapters.ChangeInformation;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class OntologicalRequirementEnforceRequest extends WriteRequest {

	private Collection<Resource> creates;
	private Collection<Resource> modis;
	private Collection<Resource> ids;
	private String author;
	private long time;

	public OntologicalRequirementEnforceRequest(Collection<Resource> creates, Collection<Resource> modis, Collection<Resource> ids) {
		this(creates,
				modis,
				ids,
				System.getProperty("user.name", ""),
				System.currentTimeMillis());
	}

	public OntologicalRequirementEnforceRequest(Collection<Resource> creates, Collection<Resource> modis, Collection<Resource> ids, String author, long time) {
		this.creates = creates;
		this.modis = modis;
		this.ids = ids;
		this.author = author;
		this.time = time;
	}

	@Override
	public void perform(WriteGraph graph) throws DatabaseException {
		update(graph, creates, modis, ids, true, author, time, true);
	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			boolean disableDependencyIndexing) throws DatabaseException
	{
		update(graph,
				creates,
				modis,
				ids,
				addComment,
				System.getProperty("user.name", ""),
				System.currentTimeMillis(),
				disableDependencyIndexing);

	}

	public static void update(
			WriteGraph graph,
			Collection<Resource> creates,
			Collection<Resource> modis,
			Collection<Resource> ids,
			boolean addComment,
			String author,
			long time,
			boolean disableDependencyIndexing) throws DatabaseException
	{
		if (disableDependencyIndexing)
			Layer0Utils.setDependenciesIndexingDisabled(graph, true);
		
		ModelingResources MOD = ModelingResources.getInstance(graph);
		Layer0 L0 = Layer0.getInstance(graph);

		if (!creates.isEmpty()) {
			ChangeInformation info = new ChangeInformation();
			info.createdAt = time;
			info.createdBy = author;
			info.modifiedAt = time;
			info.modifiedBy = author;

			for (Resource c : creates) {
				CommonDBUtils.selectClusterSet(graph, c);
				graph.claimLiteral(c, MOD.changeInformation, MOD.changeInformation_Inverse, MOD.ChangeInformation, info, ChangeInformation.BINDING);
			}
		}

		for (Resource m : modis) {
			ChangeInformation info = graph.getPossibleRelatedValue(m, MOD.changeInformation, ChangeInformation.BINDING);
			if (info == null) {
				// Should not be possible but lets handle this anyway
				info = new ChangeInformation();
				info.createdAt = time;
				info.createdBy = author;
			}
			info.modifiedAt = time;
			info.modifiedBy = author;
			CommonDBUtils.selectClusterSet(graph, m);
			graph.claimLiteral(m, MOD.changeInformation, MOD.changeInformation_Inverse, MOD.ChangeInformation, info, ChangeInformation.BINDING);
		}
		
		for (Resource r : ids) {
			if (!graph.hasStatement(r, L0.identifier)) {
				CommonDBUtils.selectClusterSet(graph, r);
				Layer0Utils.claimNewIdentifier(graph, r, true);
			}
		}

		graph.addMetadata( graph.getMetadata(CommentMetadata.class).add("Updated change information") );
		
		graph.addMetadata( graph.getMetadata(ChangeHistoryUpdated.class) );
		
	}

}