package org.simantics.modeling.scl;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationContext;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationRequest;
import org.simantics.db.layer0.util.RuntimeEnvironmentRequest;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ComponentTypeSubstructure;
import org.simantics.modeling.scl.CompileSCLMonitorRequest.CompilationContext;
import org.simantics.scl.compiler.common.names.Name;
import org.simantics.scl.compiler.constants.StringConstant;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.environment.Environments;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.top.SCLExpressionCompilationException;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CompileSCLMonitorRequest extends AbstractExpressionCompilationRequest<CompilationContext, Variable> {

    private static Logger LOGGER = LoggerFactory.getLogger(CompileSCLMonitorRequest.class);

    protected static Name BROWSE = Name.create("Simantics/Variables", "browse");
    protected static Name VALUE = Name.create("Simantics/Variables", "value");
    
    private final Resource componentType;
    private final Resource literal;
    private final Resource relation;
    
    public static class CompilationContext extends AbstractExpressionCompilationContext {
        public final ComponentTypeSubstructure substructure;
        
        public CompilationContext(RuntimeEnvironment runtimeEnvironment,
                ComponentTypeSubstructure substructure) {
            super(runtimeEnvironment);
            this.substructure = substructure;
        }
    }
    
    private CompileSCLMonitorRequest(Resource componentType, Resource literal, Resource relation) {
        this.componentType = componentType;
        this.literal = literal;
        this.relation = relation;
    }
    
    public CompileSCLMonitorRequest(ReadGraph graph, Variable context)
            throws DatabaseException {
        this(context.getParent(graph).getType(graph),
                context.getRepresents(graph),
                context.getPredicateResource(graph));
    }

    public static Object compileAndEvaluate(ReadGraph graph, Variable context) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
            Function1<Variable,Object> exp = graph.syncRequest(new CompileSCLMonitorRequest(graph, context),
                    TransientCacheListener.<Function1<Variable,Object>>instance());
            sclContext.put("graph", graph);
            return exp.apply(context.getParent(graph));
        } catch (DatabaseException e) {
            throw (DatabaseException)e;
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    @Override
    protected String getExpressionText(ReadGraph graph)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        return graph.getRelatedValue(literal, L0.SCLValue_expression, Bindings.STRING);
    }
    
    @Override
    protected Type getExpectedType(ReadGraph graph, CompilationContext context)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String valueType = graph.getPossibleRelatedValue(relation, L0.RequiresValueType, Bindings.STRING);
        if(valueType != null) {
            try {
                return Environments.getType(context.runtimeEnvironment.getEnvironment(), valueType);
            } catch (SCLExpressionCompilationException e) {
                LOGGER.warn("getExpectedType failed for valueType: " + valueType, e);
            }
        }
        return super.getExpectedType(graph, context);
    }
    
    @Override
    protected CompilationContext getCompilationContext(ReadGraph graph)
            throws DatabaseException {
        return graph.syncRequest(new ResourceRead<CompilationContext>(componentType) {
            @Override
            public CompilationContext perform(ReadGraph graph)
                    throws DatabaseException {
                Resource indexRoot = graph.syncRequest(new IndexRoot(resource));
                RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest(indexRoot));
                return new CompilationContext(runtimeEnvironment, ComponentTypeSubstructure.forType(graph, resource));
            }
        });
    }

    @Override
    protected Type getContextVariableType() {
        return VARIABLE;
    }

    @Override
    protected Expression getVariableAccessExpression(
            ReadGraph graph,
            CompilationContext context,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable,
            String name) throws DatabaseException {
        Pair<String,Type> entry = context.substructure.possibleTypedRVI(name);
        if(entry == null)
            return null;
        Environment environment = context.runtimeEnvironment.getEnvironment();
        Expression propertyVariable = new EApply(
                new EConstant(environment.getValue(BROWSE)),
                new EVariable(contextVariable),
                new ELiteral(new StringConstant(entry.first))
                );
        return makeTypeFlexible(environment, new EApply(
                new EConstant(environment.getValue(VALUE), entry.second),
                propertyVariable
                ), entry.second);
    }
    
    @Override
    public int hashCode() {
        return 31*(31*getClass().hashCode() + literal.hashCode()) + componentType.hashCode();
    }
    
    @Override
    public boolean equals(Object obj) {
        if(this == obj)
            return true;
        if(obj == null || obj.getClass() != getClass())
            return false;
        CompileSCLMonitorRequest other = (CompileSCLMonitorRequest)obj;
        return literal.equals(other.literal) && componentType.equals(other.componentType);
    }
}
