/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.requests;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.utils.page.PageDesc;
import org.simantics.utils.strings.AlphanumComparator;

/**
 * @author Tuukka Lehtonen
 */
public class Node implements Comparable<Node> {

    public static final Comparator<Node> CASE_INSENSITIVE_COMPARATOR =
            (o1, o2) -> AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(o1.getName(), o2.getName());

    private final Node       parent;
    private final List<Node> children = new ArrayList<Node>();
    private final List<Node> unmodifiableChildren = Collections.unmodifiableList(children);

    /**
     * May be <code>null</code> if there is no diagram for this node.
     */
    private final Resource         diagram;
    private final ResourceArray    definingResource; // i.e. Composite
    private final String           name;

    private PageDesc         pageDesc;
    private String           rvi;

    /**
     * @param parent
     * @param name
     * @param diagram may be <code>null</code> if there is no diagram for this node
     * @param definingResources
     */
    public Node(Node parent, String name, Resource diagram, Resource... definingResources) {
        if (definingResources.length == 0)
            throw new IllegalArgumentException("must provide at least one defining resource");
        this.parent = parent;
        this.name = name;
        this.diagram = diagram;
        this.definingResource = new ResourceArray(definingResources);

        if (parent != null)
            parent.addChild(this);
    }

    public Node cloneWithoutChildren(Node parent) {
        Node clone = new Node(parent, name, diagram, definingResource.resources);
        clone.setRVI(rvi);
        clone.setPageDesc(pageDesc);
        return clone;
    }

    public Node getParent() {
        return parent;
    }

    /**
     * @return <code>null</code> if there is no diagram for this node
     */
    public Resource getDiagramResource() {
        return diagram;
    }

    public ResourceArray getDefiningResources() {
        return definingResource;
    }

    public String getName() {
        return name;
    }

    @Override
    public String toString() {
        return (parent != null ? parent : "") + "/" + name + definingResource;
    }

    public void addChild(Node child) {
        children.add(child);
    }

    public void removeChild(Node child) {
        children.remove(child);
    }

    public Collection<Node> getChildren() {
        return unmodifiableChildren;
    }

    public void setPageDesc(PageDesc pageDesc) {
        this.pageDesc = pageDesc;
    }
    
    public void setRVI(String rvi) {
    	this.rvi = rvi;
    }
    
    public String getRVI() {
    	return rvi;
    }

    public PageDesc getPageDesc() {
        return pageDesc;
    }

    @Override
    public int compareTo(Node o) {
        int ret = name.compareTo(o.name);
        return ret;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((diagram == null) ? 0 : diagram.hashCode());
        result = prime * result + ((parent == null) ? 0 : parent.hashCode());
        result = prime * result + ((definingResource == null) ? 0 : definingResource.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Node other = (Node) obj;
        if (diagram == null) {
            if (other.diagram != null)
                return false;
        } else if (!diagram.equals(other.diagram))
            return false;
        if (parent == null) {
            if (other.parent != null)
                return false;
        } else if (!parent.equals(other.parent))
            return false;
        if (definingResource == null) {
            if (other.definingResource != null)
                return false;
        } else if (!definingResource.equals(other.definingResource))
            return false;
        return true;
    }

}