package org.simantics.scl.compiler.internal.elaboration.constraints;

import java.util.ArrayList;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.constants.ClassConstant;
import org.simantics.scl.compiler.constants.StringConstant;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.java.Builtins;
import org.simantics.scl.compiler.elaboration.modules.TypeClass;
import org.simantics.scl.compiler.elaboration.modules.TypeClassInstance;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.types.TApply;
import org.simantics.scl.compiler.types.TCon;
import org.simantics.scl.compiler.types.TFun;
import org.simantics.scl.compiler.types.TPred;
import org.simantics.scl.compiler.types.TUnion;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;

import gnu.trove.map.hash.THashMap;

public class ConstraintEnvironment {
    Environment environment;
    
    public ConstraintEnvironment(Environment environment) {
        this.environment = environment;
    }

    public Superconstraint[] getSuperconstraints(TPred constraint) {
        TypeClass tc = environment.getTypeClass(constraint.typeClass);
        if(tc == null) {
            throw new InternalCompilerError("Didn't find constraint " + constraint + ". Maybe Prelude is not loaded?");
        }
        if(tc.context.length == 0)
            return Superconstraint.EMPTY_ARRAY;
        Superconstraint[] result = new Superconstraint[tc.context.length];
        for(int i=0;i<result.length;++i) {
            result[i] = new Superconstraint(
                    (TPred)tc.context[i].replace(tc.parameters, constraint.parameters),
                    tc.superGenerators[i]
                    );
        }
        return result;
    }
	
    public Reduction reduce(TPred constraint) {
        // VecComp
        if(constraint.typeClass == Types.VEC_COMP) {
            Type parameter = Types.canonical(constraint.parameters[0]);
            if(parameter.isGround())
                return new Reduction(new ELiteral(new ClassConstant(Types.pred(Types.VEC_COMP, parameter), parameter)),
                        Type.EMPTY_ARRAY, TPred.EMPTY_ARRAY);
        }
        // Serializable
        if(constraint.typeClass == Types.SERIALIZABLE)
            return ReduceSerializable.reduceSerializable(constraint.parameters[0]);
        
        // Typeable
        else if(constraint.typeClass == Types.TYPEABLE) {
            Type parameter = Types.canonical(constraint.parameters[0]);
            if(parameter instanceof TCon) {
                TCon con = (TCon)parameter;                        
                return new Reduction(
                        new EApply(Locations.NO_LOCATION,
                                new EConstant(Builtins.INSTANCE.getValue("TCon")),
                                new ELiteral(new StringConstant(con.module)),
                                new ELiteral(new StringConstant(con.name))),
                                Type.EMPTY_ARRAY, TPred.EMPTY_ARRAY);
            }
            else if(parameter instanceof TApply) {
                TApply apply = (TApply)parameter;
                return new Reduction(
                        new EConstant(Builtins.INSTANCE.getValue("TApply")),
                        Type.EMPTY_ARRAY, new TPred[] {
                            Types.pred(Types.TYPEABLE, apply.function),
                            Types.pred(Types.TYPEABLE, apply.parameter),
                        });
            }
            else if(parameter instanceof TFun) {
                TFun fun = (TFun)parameter;
                return new Reduction(
                        new EConstant(Builtins.INSTANCE.getValue("TFun")),
                        Type.EMPTY_ARRAY, new TPred[] {
                            Types.pred(Types.TYPEABLE, fun.domain),
                            Types.pred(Types.TYPEABLE, fun.effect),
                            Types.pred(Types.TYPEABLE, fun.range)
                        });
            }
            else if(parameter instanceof TUnion) {
                TUnion union = (TUnion)parameter;
                
                if(union.effects.length == 0)
                    return new Reduction(
                            new EConstant(Builtins.INSTANCE.getValue("TPure")),
                            Type.EMPTY_ARRAY,
                            TPred.EMPTY_ARRAY);
                else if(union.effects.length == 2) {
                    return new Reduction(
                            new EConstant(Builtins.INSTANCE.getValue("TUnion2")),
                            Type.EMPTY_ARRAY, new TPred[] {
                                Types.pred(Types.TYPEABLE, union.effects[0]),
                                Types.pred(Types.TYPEABLE, union.effects[1])
                            });
                }
            }
        }
        
        // Standard case
        THashMap<TVar, Type> substitution = new THashMap<TVar, Type>();
        ArrayList<Reduction> reductions = new ArrayList<Reduction>(1); 
        for(TypeClassInstance inst : environment.getInstances(constraint.typeClass)) {
            if(Types.match(inst.instance, constraint, substitution)) {
                TPred[] demands = new TPred[inst.context.length];
                for(int i=0;i<demands.length;++i) {
                    demands[i] = (TPred)inst.context[i].replace(substitution);
                }
                Type[] parameters = new Type[inst.generatorParameters.length];
                for(int i=0;i<parameters.length;++i) {
                    Type parameter = substitution.get(inst.generatorParameters[i]);
                    if(parameter == null)
                        parameter = inst.generatorParameters[i]; // TODO Is this correct?
                    parameters[i] = parameter;
                }
                reductions.add(new Reduction(new ELiteral(inst.generator), parameters, demands));
            }
            substitution.clear();
        }
        //System.out.println(constraint.typeClass + " -> " + reductions.size());
        if(reductions.size() == 1)
            return reductions.get(0);
        else if(reductions.size() > 1) {
            throw new InternalCompilerError("Found more than one matching instances for " + constraint.typeClass + ".");
        }
        return null;
    }
}
