package org.simantics.scl.compiler.elaboration.java;

import org.cojen.classfile.TypeDesc;
import org.osgi.service.component.annotations.Component;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.compiler.common.names.Name;
import org.simantics.scl.compiler.constants.JavaMethod;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EExternalConstant;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.macros.MacroRule;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.internal.codegen.types.StandardTypeConstructor;
import org.simantics.scl.compiler.module.ConcreteModule;
import org.simantics.scl.compiler.types.TCon;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component
public class LoggingModule extends ConcreteModule {

    private static String[] LOGGING_METHODS = new String[] {
            "trace", "debug", "info", "warn", "error" 
    };

    public LoggingModule() {
        super("Logging");

        // Logger
        TCon Logger = Types.con(getName(), "Logger");
        StandardTypeConstructor loggerConstructor = new StandardTypeConstructor(Logger, Kinds.STAR, TypeDesc.forClass(Logger.class));
        loggerConstructor.external = true;
        addTypeDescriptor("Logger", loggerConstructor);

        // Common types
        Type loggingType = Types.functionE(Types.STRING, Types.PROC, Types.UNIT);

        // Add logging methods
        for(String methodName : LOGGING_METHODS) {
            JavaMethod javaMethod = new JavaMethod(false, "org/slf4j/Logger", methodName, Types.PROC, Types.UNIT, Logger, Types.STRING);
            SCLValue value = new SCLValue(Name.create(getName(), methodName));
            value.setType(loggingType);
            value.setMacroRule(new MacroRule() {
                @Override
                public Expression apply(SimplificationContext context, Type[] typeParameters, EApply apply) {
                    ConcreteModule module = context.getCompilationContext().module;
                    String identifier;
                    if (module != null)
                        identifier = module.getName().replaceAll("/", ".");
                    else
                        identifier = CommandSession.class.getName();
                    apply.set(new ELiteral(javaMethod), new Expression[] {
                            new EExternalConstant(LoggerFactory.getLogger(identifier), Logger),
                            apply.parameters[0]
                    });
                    return apply;
                }
            });
            addValue(value);
        }
        
        setParentClassLoader(LoggerFactory.class.getClassLoader());
    }

}
