package org.simantics.scl.compiler.elaboration.java;


import org.simantics.scl.compiler.constants.DoubleConstant;
import org.simantics.scl.compiler.constants.FloatConstant;
import org.simantics.scl.compiler.constants.FunctionValue;
import org.simantics.scl.compiler.constants.IntegerConstant;
import org.simantics.scl.compiler.constants.JavaComparisonOperation;
import org.simantics.scl.compiler.constants.JavaComparisonToZeroOperation;
import org.simantics.scl.compiler.constants.JavaConversionOperation;
import org.simantics.scl.compiler.constants.JavaMathOperation;
import org.simantics.scl.compiler.constants.JavaMethod;
import org.simantics.scl.compiler.constants.LongConstant;
import org.simantics.scl.compiler.constants.ShortConstant;
import org.simantics.scl.compiler.constants.singletons.UnsafeCoerce;
import org.simantics.scl.compiler.elaboration.expressions.EApplyType;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.module.ConcreteModule;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;

public class JavaModule extends ConcreteModule {

    public static final JavaModule INSTANCE = new JavaModule();
    
    public static final String MODULE_NAME = "JavaBuiltin";

    private JavaModule() {
        super(MODULE_NAME);
        addValue("dconst_0", new DoubleConstant(0.0));
        addValue("dconst_1", new DoubleConstant(1.0));
        addValue("fconst_0", new FloatConstant(0.0f));
        addValue("fconst_1", new FloatConstant(1.0f));
        addValue("iconst_0", IntegerConstant.ZERO);
        addValue("iconst_1", IntegerConstant.ONE);
        addValue("sconst_0", ShortConstant.ZERO);
        addValue("sconst_1", ShortConstant.ONE);
        addValue("lconst_0", new LongConstant(0L));
        addValue("lconst_1", new LongConstant(1L));
        
        // Mathematical operations
        for(JavaMathOperation operation : JavaMathOperation.OPCODES)
            addValue(operation.getMnemonic(), operation);
        addValue("cadd", JavaMathOperation.CADD);
        addValue("csub", JavaMathOperation.CSUB);
        
        addValue("sadd", JavaMathOperation.SADD);
        addValue("ssub", JavaMathOperation.SSUB);
        addValue("smul", JavaMathOperation.SMUL);
        addValue("sdiv", JavaMathOperation.SDIV);
        addValue("srem", JavaMathOperation.SREM);
        addValue("sneg", JavaMathOperation.SNEG);
        addValue("sand", JavaMathOperation.SAND);
        addValue("sor", JavaMathOperation.SOR);
        addValue("sxor", JavaMathOperation.SXOR);
        
        addValue("badd", JavaMathOperation.BADD);
        addValue("bsub", JavaMathOperation.BSUB);
        addValue("bmul", JavaMathOperation.BMUL);
        addValue("bdiv", JavaMathOperation.BDIV);
        addValue("brem", JavaMathOperation.BREM);
        addValue("bneg", JavaMathOperation.BNEG);
        addValue("band", JavaMathOperation.BAND);
        addValue("bor", JavaMathOperation.BOR);
        addValue("bxor", JavaMathOperation.BXOR);
        
        // Conversions
        for(JavaConversionOperation operation : JavaConversionOperation.OPCODES)
            addValue(operation.getMnemonic(), operation);
        
        // Comparison operations
        addValue("icmpeq", new JavaComparisonOperation("==", Types.INTEGER));
        addValue("icmpne", new JavaComparisonOperation("!=", Types.INTEGER));
        addValue("icmplt", new JavaComparisonOperation("<", Types.INTEGER));
        addValue("icmple", new JavaComparisonOperation("<=", Types.INTEGER));
        addValue("icmpgt", new JavaComparisonOperation(">", Types.INTEGER));
        addValue("icmpge", new JavaComparisonOperation(">=", Types.INTEGER));
        
        addValue("lcmpeq", new JavaComparisonOperation("==", Types.LONG));
        addValue("lcmpne", new JavaComparisonOperation("!=", Types.LONG));
        addValue("lcmplt", new JavaComparisonOperation("<", Types.LONG));
        addValue("lcmple", new JavaComparisonOperation("<=", Types.LONG));
        addValue("lcmpgt", new JavaComparisonOperation(">", Types.LONG));
        addValue("lcmpge", new JavaComparisonOperation(">=", Types.LONG));     
        
        addValue("bcmpeq", new JavaComparisonOperation("==", Types.BYTE));
        addValue("bcmpne", new JavaComparisonOperation("!=", Types.BYTE));
        addValue("bcmplt", new JavaComparisonOperation("<", Types.BYTE));
        addValue("bcmple", new JavaComparisonOperation("<=", Types.BYTE));
        addValue("bcmpgt", new JavaComparisonOperation(">", Types.BYTE));
        addValue("bcmpge", new JavaComparisonOperation(">=", Types.BYTE));
        
        addValue("scmpeq", new JavaComparisonOperation("==", Types.SHORT));
        addValue("scmpne", new JavaComparisonOperation("!=", Types.SHORT));
        addValue("scmplt", new JavaComparisonOperation("<", Types.SHORT));
        addValue("scmple", new JavaComparisonOperation("<=", Types.SHORT));
        addValue("scmpgt", new JavaComparisonOperation(">", Types.SHORT));
        addValue("scmpge", new JavaComparisonOperation(">=", Types.SHORT));
        
        addValue("fcmpeq", new JavaComparisonOperation("==", Types.FLOAT));
        addValue("fcmpne", new JavaComparisonOperation("!=", Types.FLOAT));
        addValue("fcmplt", new JavaComparisonOperation("<", Types.FLOAT));
        addValue("fcmple", new JavaComparisonOperation("<=", Types.FLOAT));
        addValue("fcmpgt", new JavaComparisonOperation(">", Types.FLOAT));
        addValue("fcmpge", new JavaComparisonOperation(">=", Types.FLOAT));    
        
        addValue("dcmpeq", new JavaComparisonOperation("==", Types.DOUBLE));
        addValue("dcmpne", new JavaComparisonOperation("!=", Types.DOUBLE));
        addValue("dcmplt", new JavaComparisonOperation("<", Types.DOUBLE));
        addValue("dcmple", new JavaComparisonOperation("<=", Types.DOUBLE));
        addValue("dcmpgt", new JavaComparisonOperation(">", Types.DOUBLE));
        addValue("dcmpge", new JavaComparisonOperation(">=", Types.DOUBLE));        

        addValue("ccmpeq", new JavaComparisonOperation("==", Types.CHARACTER));
        addValue("ccmpne", new JavaComparisonOperation("!=", Types.CHARACTER));
        addValue("ccmplt", new JavaComparisonOperation("<", Types.CHARACTER));
        addValue("ccmple", new JavaComparisonOperation("<=", Types.CHARACTER));
        addValue("ccmpgt", new JavaComparisonOperation(">", Types.CHARACTER));
        addValue("ccmpge", new JavaComparisonOperation(">=", Types.CHARACTER));
        
        // Comparison to zero operations
        addValue("ifeq", new JavaComparisonToZeroOperation("=="));
        addValue("ifne", new JavaComparisonToZeroOperation("!="));
        addValue("iflt", new JavaComparisonToZeroOperation("<"));
        addValue("ifle", new JavaComparisonToZeroOperation("<="));
        addValue("ifgt", new JavaComparisonToZeroOperation(">"));
        addValue("ifge", new JavaComparisonToZeroOperation(">="));
        
        TVar A = Types.var(Kinds.STAR);
        
        addValue("unsafeCoerce", UnsafeCoerce.INSTANCE);
        addValue("equals", new JavaMethod(true, 
                "java/lang/Object", "equals", Types.NO_EFFECTS, Types.BOOLEAN, A, A));
        addValue("hashCode", new JavaMethod(true, 
                "java/lang/Object", "hashCode", Types.NO_EFFECTS, Types.INTEGER, A));
        addValue("toString", new JavaMethod(true, 
                "java/lang/Object", "toString", Types.NO_EFFECTS, Types.STRING, A));
        
        setParentClassLoader(getClass().getClassLoader());
    }
    
    static Expression createLiteral(FunctionValue value) {
        Expression result = new ELiteral(value);
        for(TVar var : value.getTypeParameters())
            result = new EApplyType(result, var);
        return result;
    }
    
}

