package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.continuations.ICont;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.ssa.exits.Throw;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class GuardedExpressionGroup extends Expression {
    public GuardedExpression[] expressions;

    public GuardedExpressionGroup(GuardedExpression[] expressions) {
        this.expressions = expressions;
    }

    @Override
    public void collectRefs(TObjectIntHashMap<Object> allRefs,
            TIntHashSet refs) {
        for(GuardedExpression expression : expressions) {
            for(Expression guard : expression.guards)
                guard.collectRefs(allRefs, refs);
            expression.value.collectRefs(allRefs, refs);
        }
    }

    @Override
    public void collectVars(TObjectIntHashMap<Variable> allVars,
            TIntHashSet vars) {
        for(GuardedExpression expression : expressions) {
            for(Expression guard : expression.guards)
                guard.collectVars(allVars, vars);
            expression.value.collectVars(allVars, vars);
        }
    }

    @Override
    protected void updateType() throws MatchException {
        setType(expressions[0].value.getType());
    }

    @Override
    public IVal toVal(Environment env, CodeWriter w) {
        CodeWriter success = w.createBlock(getType());
        IVal result = success.getParameters()[0];
        CodeWriter failure = w.createBlock();
        compile(env, w, success.getContinuation(), failure.getContinuation());
        w.continueAs(success);
        failure.throw_(location, Throw.MatchingException, "Matching failure at: " + toString());
        return result;
        //throw new InternalCompilerError("GuardedExpressionGroup should be handled in match compilation.");
    }

    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
        for(GuardedExpression expression : expressions) {
            for(Expression guard : expression.guards)
                guard.collectFreeVariables(vars);
            expression.value.collectFreeVariables(vars);
        }
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        for(GuardedExpression expression : expressions) {
            for(int i=0;i<expression.guards.length;++i)
                expression.guards[i] = expression.guards[i].simplify(context);
            expression.value = expression.value.simplify(context);
        }
        return this;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        for(GuardedExpression expression : expressions) {
            for(int i=0;i<expression.guards.length;++i)
                expression.guards[i] = expression.guards[i].resolve(context);
            expression.value = expression.value.resolve(context);
        }
        return this;
    }
    
    @Override
    public Expression checkBasicType(TypingContext context, Type requiredType) {
        for(GuardedExpression expression : expressions) {
            for(int i=0;i<expression.guards.length;++i)
                expression.guards[i] = expression.guards[i].checkType(context, Types.BOOLEAN);
            expression.value = expression.value.checkType(context, requiredType);
        }
        return this;
    }
    
    public void compile(Environment env, CodeWriter firstWriter, ICont success,
            ICont lastFailure) {
        // Create all code blocks
        CodeWriter[] writers = new CodeWriter[expressions.length];                
        ICont[] failures = new ICont[expressions.length];
        writers[0] = firstWriter;
        failures[expressions.length-1] = lastFailure;
        for(int i=1;i<expressions.length;++i) {
            CodeWriter writer = firstWriter.createBlock();
            writers[i] = writer;
            failures[i-1] = writer.getContinuation();
        }
        
        // Compile
        for(int i=0;i<expressions.length;++i) {
            CodeWriter w = writers[i];
            ICont failure = failures[i];
            
            for(Expression guard : expressions[i].guards) {
                CodeWriter nextW = w.createBlock();
                w.if_(guard.toVal(env, w), nextW.getContinuation(), failure);
                w = nextW;
            }
            
            w.jump(success, expressions[i].value.toVal(env, w));
        }
    }
    
    @Override
    public Expression replace(ReplaceContext context) {
        GuardedExpression[] newExpressions = new GuardedExpression[expressions.length];
        for(int i=0;i<expressions.length;++i)
            newExpressions[i] = expressions[i].replace(context);
        return new GuardedExpressionGroup(newExpressions);            
    }

    @Override
    public Expression decorate(ExpressionDecorator decorator) {
        for(GuardedExpression expression : expressions)
            expression.decorate(decorator);
        return decorator.decorate(this);
    }

    @Override
    public void collectEffects(THashSet<Type> effects) {
        for(GuardedExpression ge : expressions) {
            for(Expression guard : ge.guards)
                guard.collectEffects(effects);
            ge.value.collectEffects(effects);
        }
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            for(GuardedExpression expression : expressions)
                expression.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public void forVariables(VariableProcedure procedure) {
        for(GuardedExpression expression : expressions)
            expression.forVariables(procedure);
    }

    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public int getSyntacticFunctionArity() {
        return expressions[0].value.getSyntacticFunctionArity();
    }
}
