package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.java.Builtins;
import org.simantics.scl.compiler.elaboration.java.ListConstructor;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.utils.Constants;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.internal.interpreted.IListLiteral;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class EListLiteral extends SimplifiableExpression {

    Expression[] components;
    Type componentType;

    public EListLiteral(Expression[] components) {
        this.components = components;
    }

    private EListLiteral(Expression[] components, Type componentType) {
        this.components = components;
        this.componentType = componentType;
    }

    public Expression[] getComponents() {
        return components;
    }
    
    @Override
    public void collectRefs(TObjectIntHashMap<Object> allRefs,
            TIntHashSet refs) {
        for(Expression component : components)
            component.collectRefs(allRefs, refs);
    }

    @Override
    public void collectVars(TObjectIntHashMap<Variable> allVars,
            TIntHashSet vars) {
        for(Expression component : components)
            component.collectVars(allVars, vars);
    }
    
    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
        for(Expression component : components)
            component.collectFreeVariables(vars);
    }
    
    @Override
    public Expression simplify(SimplificationContext context) {
        context.pushLocation(location);
        try {
            for(int i=0;i<components.length;++i)
                components[i] = components[i].simplify(context);
            
            if(components.length <= Constants.MAX_LIST_LITERAL_LENGTH) {
                Expression result = new EConstant(location, Builtins.LIST_CONSTRUCTORS[components.length], 
                        componentType);
                if(components.length > 0)
                    result = new EApply(location, result, components);
                return result;
            }
            else {
                Expression result = new EApplyType(new ELiteral(location, new ListConstructor(components.length)), componentType);
                result = new EApply(location, result, components);
                return result;
            }
        } finally {
            context.popLocation();
        }
    }

    @Override
    public Expression resolve(TranslationContext context) {
        for(int i=0;i<components.length;++i)
            components[i] = components[i].resolve(context);
        return this;
    }
    
    @Override
    public Expression resolveAsPattern(TranslationContext context) {
        for(int i=0;i<components.length;++i)
            components[i] = components[i].resolveAsPattern(context);
        return this;
    }
    
    @Override
    protected void updateType() throws MatchException {
        setType(Types.list(componentType));
    }
    
    @Override
    public Expression checkBasicType(TypingContext context, Type requiredType) {
        try {
            componentType = Types.unifyApply(Types.LIST, requiredType);
        } catch (MatchException e) {
            context.getErrorLog().log(location, "Expected a value with type " + requiredType + " but got a list.");
            return new EError(location);
        }
        for(int i=0;i<components.length;++i)
            components[i] = components[i].checkType(context, componentType);
        return this;
    }

    @Override
    public Expression decorate(ExpressionDecorator decorator) {
        for(int i=0;i<components.length;++i)
            components[i] = components[i].decorate(decorator);
        return decorator.decorate(this);
    }

    @Override
    public void collectEffects(THashSet<Type> effects) {
        for(Expression component : components)
            component.collectEffects(effects);
    }

    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            for(Expression component : components)
                component.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext target) {
        IExpression[] componentExpressions = new IExpression[components.length];
        for(int i=0;i<components.length;++i)
            componentExpressions[i] = components[i].toIExpression(target);
        return new IListLiteral(componentExpressions);
    }

    @Override
    public void forVariables(VariableProcedure procedure) {
        for(Expression component : components)
            component.forVariables(procedure);
    }
    
    @Override
    public Expression replace(ReplaceContext context) {
        Expression[] newComponents = new Expression[components.length];
        for(int i=0;i<components.length;++i)
            newComponents[i] = components[i].replace(context);
        return new EListLiteral(newComponents, componentType.replace(context.tvarMap));
    }
    
    @Override
    public boolean isPattern(int arity) {
        if(arity != 0)
            return false;
        for(Expression component : components)
            if(!component.isPattern(0))
                return false;
        return true;
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }

}
