package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.chr.CHRRuleset;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.exceptions.MatchException;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class ECHRRuleset extends Expression {
    CHRRuleset ruleset;
    Expression in;
    
    public ECHRRuleset(CHRRuleset ruleset, Expression in) {
        this.ruleset = ruleset;
        this.in = in;
    }
    
    @Override
    public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        ruleset.collectRefs(allRefs, refs);
        in.collectRefs(allRefs, refs);
    }
    @Override
    public void collectVars(TObjectIntHashMap<Variable> allVars, TIntHashSet vars) {
        ruleset.collectVars(allVars, vars);
        in.collectVars(allVars, vars);
    }
    @Override
    public void forVariables(VariableProcedure procedure) {
        ruleset.forVariables(procedure);
        in.forVariables(procedure);
    }
    @Override
    protected void updateType() throws MatchException {
        setType(in.getType());
    }
    @Override
    public IVal toVal(Environment env, CodeWriter w) {
        ruleset.generateCode(w);
        return in.toVal(env, w);
    }
    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
        ruleset.collectFreeVariables(vars);
        in.collectFreeVariables(vars);
    }
    @Override
    public Expression resolve(TranslationContext context) {
        context.pushFrame();
        context.pushCHRConstraintFrame();
        ruleset.resolve(context);
        in = in.resolve(context);
        context.popCHRConstraintFrame(ruleset.constraints);
        context.popFrame();
        return this;
    }
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            this.location = loc;
            ruleset.setLocationDeep(loc);
            in.setLocationDeep(loc);
        }
    }
    @Override
    public Expression decorate(ExpressionDecorator decorator) {
        in = in.decorate(decorator);
        return this;
    }
    @Override
    public void collectEffects(THashSet<Type> effects) {
        ruleset.collectEffects(effects);
        in.collectEffects(effects);
    }
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public Expression inferType(TypingContext context) {
        ruleset.checkType(context);
        in = in.inferType(context);
        return this;
    }
    
    @Override
    public Expression simplify(SimplificationContext context) {
        ruleset.simplify(context);
        ruleset.compile(context);
        in = in.simplify(context);
        return this;
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext context) {
        throw new UnsupportedOperationException();
    }

}
