package org.simantics.scl.compiler.elaboration.chr;

import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.chr.plan.PlanOp;
import org.simantics.scl.compiler.elaboration.chr.plan.PrioritizedPlan;
import org.simantics.scl.compiler.elaboration.chr.planning.QueryPlanningContext;
import org.simantics.scl.compiler.elaboration.chr.relations.CHRConstraint;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Variable;
import org.simantics.scl.compiler.elaboration.expressions.VariableProcedure;
import org.simantics.scl.compiler.elaboration.expressions.printing.ExpressionToStringVisitor;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.parsing.Symbol;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class CHRRule extends Symbol {
    public int priority;
    public CHRQuery head;
    public CHRQuery body;
    public Variable[] existentialVariables;
    
    // Analysis
    public int firstPriorityExecuted;
    public int lastPriorityExecuted;
    
    public CHRRule(long location, CHRQuery head, CHRQuery body, Variable[] existentialVariables) {
        this.location = location;
        this.head = head;
        this.body = body;
        this.existentialVariables = existentialVariables;
    }

    public void resolve(TranslationContext context) {
        context.pushExistentialFrame();
        head.resolve(context);
        body.resolve(context);
        existentialVariables = context.popExistentialFrame();
    }

    public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        head.collectRefs(allRefs, refs);
        body.collectRefs(allRefs, refs);
    }

    public void checkType(TypingContext context) {
        for(Variable variable : existentialVariables)
            variable.setType(Types.metaVar(Kinds.STAR));
        head.checkType(context);
        body.checkType(context);
    }
    
    public void collectVars(TObjectIntHashMap<Variable> allVars, TIntHashSet vars) {
        head.collectVars(allVars, vars);
        body.collectVars(allVars, vars);
    }

    public void forVariables(VariableProcedure procedure) {
        head.forVariables(procedure);
        body.forVariables(procedure);
    }

    public void collectFreeVariables(THashSet<Variable> vars) {
        head.collectFreeVariables(vars);
        body.collectFreeVariables(vars);
    }

    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            this.location = loc;
            head.setLocationDeep(loc);
            body.setLocationDeep(loc);
        }
    }
    
    public void simplify(SimplificationContext context) {
        head.simplify(context);
        body.simplify(context);
    }

    public void compile(CompilationContext compilationContext, CHRConstraint initConstraint) {
        boolean hasActiveLiteral = false;
        for(int i=0;i<head.literals.length;++i) {
            CHRLiteral literal = head.literals[i];
            if(literal.passive)
                continue;
            CHRConstraint constraint = (CHRConstraint)literal.relation;
            
            Variable activeFact = new Variable("activeFact", constraint.factType);
            QueryPlanningContext context = new QueryPlanningContext(compilationContext, existentialVariables);
            if(!head.createQueryPlan(context, new EVariable(activeFact), i))
                return;
            body.createEnforcePlan(context, priority);
            constraint.plans.add(new PrioritizedPlan(priority, activeFact, context.getPlanOps()));
            
            hasActiveLiteral = true;
        }
        if(!hasActiveLiteral) {
            Variable activeFact = new Variable("activeFact", initConstraint.factType);
            QueryPlanningContext context = new QueryPlanningContext(compilationContext, existentialVariables);
            if(!head.createQueryPlan(context, null, -1))
                return;
            body.createEnforcePlan(context, priority);
            /*System.out.println(this);
            for(PlanOp planOp : context.getPlanOps())
                System.out.println("    " + planOp);*/
            initConstraint.plans.add(new PrioritizedPlan(priority, activeFact, context.getPlanOps()));
        }
    }
    
    public String toString() {
        StringBuilder b = new StringBuilder();
        ExpressionToStringVisitor visitor = new ExpressionToStringVisitor(b);
        visitor.visit(this);
        return b.toString();
    }
    
}
