package org.simantics.scl.compiler.elaboration.chr;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.elaboration.chr.relations.CHRConstraint;
import org.simantics.scl.compiler.elaboration.chr.relations.ExternalCHRRelation;
import org.simantics.scl.compiler.elaboration.chr.relations.SpecialCHRRelation;
import org.simantics.scl.compiler.elaboration.chr.relations.UnresolvedCHRRelation;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.expressions.Variable;
import org.simantics.scl.compiler.elaboration.expressions.VariableProcedure;
import org.simantics.scl.compiler.elaboration.expressions.printing.ExpressionToStringVisitor;
import org.simantics.scl.compiler.elaboration.relations.SCLRelation;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.parsing.Symbol;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class CHRLiteral extends Symbol {
    
    public CHRRelation relation;
    public Type[] typeParameters;
    public Expression[] parameters;
    public Expression[] typeConstraintEvidenceParameters;
    public boolean killAfterMatch;
    public boolean negated;
    public boolean passive = true;
    
    public CHRLiteral(long location, CHRRelation relation, Expression[] parameters, boolean remove, boolean negated) {
        this.location = location;
        this.relation = relation;
        this.parameters = parameters;
        this.killAfterMatch = remove;
        this.negated = negated;
    }

    public void resolve(TranslationContext context) {
        if(relation instanceof UnresolvedCHRRelation) {
            UnresolvedCHRRelation unresolved = (UnresolvedCHRRelation)relation;
            CHRConstraint constraint = context.resolveCHRConstraint(unresolved.name);
            if(constraint != null) {
                relation = constraint;
                passive = false;
            }
            else {
                SCLRelation sclRelation = context.resolveRelation(unresolved.location, unresolved.name);
                if(sclRelation != null)
                    relation = new ExternalCHRRelation(sclRelation);
                else {
                    Type[] parameterTypes = new Type[parameters.length];
                    for(int i=0;i<parameterTypes.length;++i)
                        parameterTypes[i] = Types.metaVar(Kinds.STAR);
                    constraint = new CHRConstraint(location, unresolved.name, parameterTypes);
                    constraint.implicitlyDeclared = true;
                    context.newCHRConstraint(constraint.name, constraint);
                    relation = constraint;
                    passive = false;
                    //context.getErrorLog().log(unresolved.location, "Couldn't resolve constraint " + unresolved.name + ".");
                }
            }
        }
        for(int i=0;i<parameters.length;++i)
            parameters[i] = parameters[i].resolve(context);
    }

    public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        for(Expression parameter : parameters)
            parameter.collectRefs(allRefs, refs);
        if(typeConstraintEvidenceParameters != null)
            for(Expression parameter : typeConstraintEvidenceParameters)
                parameter.collectRefs(allRefs, refs);
    }

    public void checkType(TypingContext context) {
        if(relation == SpecialCHRRelation.EXECUTE) {
            if(parameters.length != 1)
                throw new InternalCompilerError("Wrong number of parameters for EXECUTE constraint.");
            parameters[0] = parameters[0].checkIgnoredType(context);
            typeConstraintEvidenceParameters = Expression.EMPTY_ARRAY;
        }
        else {
            TVar[] typeVariables = relation.getTypeVariables();
            typeParameters = typeVariables.length == 0 ? Type.EMPTY_ARRAY : new Type[typeVariables.length];
            for(int i=0;i<typeVariables.length;++i)
                typeParameters[i] = Types.metaVar(typeVariables[i].getKind());
            Type[] parameterTypes = Types.replace(relation.getParameterTypes(), typeVariables, typeParameters);
            if(parameterTypes.length != parameters.length)
                context.getErrorLog().log(location, "Constraint is applied with wrong number of parameters");
            else
                for(int i=0;i<parameters.length;++i)
                    parameters[i] = parameters[i].checkType(context, parameterTypes[i]);
            
            typeConstraintEvidenceParameters = context.addConstraints(Types.replace(relation.getTypeConstraints(), typeVariables, typeParameters));
        }
    }

    public void collectVars(TObjectIntHashMap<Variable> allVars, TIntHashSet vars) {
        for(Expression parameter : parameters)
            parameter.collectVars(allVars, vars);
        if(typeConstraintEvidenceParameters != null)
            for(Expression parameter : typeConstraintEvidenceParameters)
                parameter.collectVars(allVars, vars);
    }

    public void forVariables(VariableProcedure procedure) {
        for(Expression parameter : parameters)
            parameter.forVariables(procedure);
        if(typeConstraintEvidenceParameters != null)
            for(Expression parameter : typeConstraintEvidenceParameters)
                parameter.forVariables(procedure);
    }

    public void collectFreeVariables(THashSet<Variable> vars) {
        for(Expression parameter : parameters)
            parameter.collectFreeVariables(vars);
        if(typeConstraintEvidenceParameters != null)
            for(Expression parameter : typeConstraintEvidenceParameters)
                parameter.collectFreeVariables(vars);
    }

    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            this.location = loc;
            for(Expression parameter : parameters)
                parameter.setLocationDeep(loc);
        }
    }
    
    public void simplify(SimplificationContext context) {
        for(int i=0;i<parameters.length;++i)
            parameters[i] = parameters[i].simplify(context);
        if(typeConstraintEvidenceParameters != null)
            for(int i=0;i<typeConstraintEvidenceParameters.length;++i)
                typeConstraintEvidenceParameters[i] = typeConstraintEvidenceParameters[i].simplify(context);
    }
    
    public String toString() {
        StringBuilder b = new StringBuilder();
        ExpressionToStringVisitor visitor = new ExpressionToStringVisitor(b);
        visitor.visit(this);
        return b.toString();
    }

    public void collectQueryEffects(THashSet<Type> effects) {
    }

    public void collectEnforceEffects(THashSet<Type> effects) {
    }
}
