package org.simantics.scl.compiler.compilation;

import java.util.Arrays;
import java.util.function.Consumer;

import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.elaboration.modules.TypeClass;
import org.simantics.scl.compiler.elaboration.modules.TypeDescriptor;
import org.simantics.scl.compiler.elaboration.relations.SCLEntityType;
import org.simantics.scl.compiler.elaboration.relations.SCLRelation;
import org.simantics.scl.compiler.elaboration.rules.MappingRelation;
import org.simantics.scl.compiler.elaboration.rules.TransformationRule;
import org.simantics.scl.compiler.environment.AmbiguousNameException;
import org.simantics.scl.compiler.environment.Namespace;
import org.simantics.scl.compiler.environment.filter.AcceptAllNamespaceFilter;
import org.simantics.scl.compiler.environment.filter.NamespaceFilter;
import org.simantics.scl.compiler.internal.codegen.effects.EffectConstructor;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.top.SCLCompilerConfiguration;
import org.simantics.scl.compiler.types.TCon;

import gnu.trove.procedure.TObjectProcedure;

public class NamespaceOfModule implements Namespace {
    private final Namespace base;
    private final Module module;
    
    public NamespaceOfModule(Namespace base, Module module) {
        this.base = base;
        this.module = module;
    }

    @Override
    public Namespace getNamespace(String name) {
        return base.getNamespace(name);
    }

    @Override
    public SCLValue getValue(String name) throws AmbiguousNameException {
        SCLValue value = module.getValue(name);
        if(SCLCompilerConfiguration.ALLOW_OVERLOADING) {
            SCLValue value2;
            try {
                value2 = base.getValue(name);
            } catch(AmbiguousNameException e) {
                if(value != null) {
                    String[] conflictingModules = Arrays.copyOf(e.conflictingModules, e.conflictingModules.length+1);
                    conflictingModules[e.conflictingModules.length] = module.getName();
                    throw new AmbiguousNameException(Arrays.asList(conflictingModules), e.name);
                }
                else
                    throw e;
            }
            if(value == null)
                return value2;
            if(value2 == null)
                return value;
            throw new AmbiguousNameException(Arrays.asList(value.getName().module, value2.getName().module), value.getName().name);
        }
        else {
            if(value != null)
                return value;
            return base.getValue(name);
        }
    }
    
    @Override
    public SCLRelation getRelation(String name) throws AmbiguousNameException {
        SCLRelation relation = module.getRelation(name);
        if(relation != null)
            return relation;
        return base.getRelation(name);
    }

    @Override
    public SCLEntityType getEntityType(String name)
            throws AmbiguousNameException {
        SCLEntityType entityType = module.getEntityType(name);
        if(entityType != null)
            return entityType;
        return base.getEntityType(name);
    }

    @Override
    public TypeDescriptor getTypeDescriptor(String name)
            throws AmbiguousNameException {
        TypeDescriptor typeDescriptor = module.getTypeDescriptor(name);
        if(typeDescriptor != null)
            return typeDescriptor;
        return base.getTypeDescriptor(name);
    }

    @Override
    public EffectConstructor getEffectConstructor(String name)
            throws AmbiguousNameException {
        EffectConstructor effectConstructor = module.getEffectConstructor(name);
        if(effectConstructor != null)
            return effectConstructor;
        return base.getEffectConstructor(name);
    }

    @Override
    public TypeClass getTypeClass(String name) throws AmbiguousNameException {
        TypeClass typeClass = module.getTypeClass(name);
        if(typeClass != null)
            return typeClass;
        return base.getTypeClass(name);
    }
    
    @Override
    public MappingRelation getMappingRelation(String name)
            throws AmbiguousNameException {
        MappingRelation mappingRelation = module.getMappingRelation(name);
        if(mappingRelation != null)
            return mappingRelation;
        return base.getMappingRelation(name);
    }
    
    @Override
    public TransformationRule getRule(String name) throws AmbiguousNameException {
        TransformationRule rule = module.getRule(name);
        if(rule != null)
            return rule;
        return base.getRule(name);
    }

    @Override
    public void findValuesForPrefix(String prefix, NamespaceFilter filter, TObjectProcedure<SCLValue> proc) {
        base.findValuesForPrefix(prefix, filter, proc);
        module.findValuesForPrefix(prefix, AcceptAllNamespaceFilter.INSTANCE, proc);
    }

    @Override
    public void findTypesForPrefix(String prefix, NamespaceFilter filter, Consumer<TCon> consumer) {
        base.findTypesForPrefix(prefix, filter, consumer);
        module.findTypesForPrefix(prefix, AcceptAllNamespaceFilter.INSTANCE, consumer);
    }
    
}
