/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.template2d.ui.wizard;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Deque;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.db.Resource;
import org.simantics.db.management.ISessionContext;
import org.simantics.modeling.template2d.DiagramTemplates;
import org.simantics.modeling.template2d.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class DrawingTemplateImportWizard extends Wizard implements IImportWizard {

    private static final int MAX_RECENT_IMPORT_PATHS = 10;

    ImportPlan        importModel;

    private boolean readPreferences(IStructuredSelection selection) {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(Preferences.RECENT_DRAWING_TEMPLATE_IMPORT_LOCATIONS);
        Deque<String> recentImportPaths = Preferences.decodePaths(recentPathsPref);

        ISessionContext ctx = SimanticsUI.getSessionContext();
        if (ctx == null)
            return false;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return false;

        importModel = new ImportPlan(ctx, recentImportPaths);
        importModel.project = project;
        importModel.selection = selection.getFirstElement();

        return true;
    }

    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.RECENT_DRAWING_TEMPLATE_IMPORT_LOCATIONS, Preferences.encodePaths(importModel.recentLocations));

        if (store.needsSaving())
            store.save();
    }

    public DrawingTemplateImportWizard() {
        setWindowTitle("Import Diagram Template");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences(selection);
    }

    @Override
    public void addPages() {
        super.addPages();
        if (importModel != null) {
            addPage(new DrawingTemplateImportPage(importModel));
        } else {
            addPage(new NoProjectPage("Import Diagram Template"));
        }
    }

    @Override
    public boolean performFinish() {
        try {
        	importModel.recentLocations.addFirst(importModel.importLocation.getAbsolutePath());
            Preferences.removeDuplicates(importModel.recentLocations);
            if (importModel.recentLocations.size() > MAX_RECENT_IMPORT_PATHS)
            	importModel.recentLocations.pollLast();

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        try {
            getContainer().run(true, true, new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        Resource target = ResourceAdaptionUtils.toSingleResource(importModel.selection);
                        DiagramTemplates.importTemplate(monitor, importModel.sessionContext.getSession(), importModel.importLocation, target);
                    } catch (Exception e) {
                        throw new InvocationTargetException(e);
                    }
                }
            });
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            if (t instanceof IOException) {
                cp.setErrorMessage("An I/O problem occurred while importing a diagram template.\n\nMessage: " + e.getMessage());
            }
            ErrorLogger.defaultLogError(t);
            return false;
        } catch (InterruptedException e) {
            ExceptionUtils.logAndShowError(e);
            return false;
        }

        return true;
    }

}
