/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.template2d.ui.wizard;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Deque;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.db.management.ISessionContext;
import org.simantics.modeling.template2d.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.ui.SimanticsUI;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 * @author Teemu Mtsniemi
 */
public class DrawingTemplateExportWizard extends Wizard implements IExportWizard {

    private static final int MAX_RECENT_EXPORT_PATHS = 10;

    Deque<String>            recentExportPaths;
    boolean                  overwrite;

    ExportPlan        exportModel;

    private boolean readPreferences() {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        String recentPathsPref = store.getString(Preferences.RECENT_DRAWING_TEMPLATE_EXPORT_LOCATIONS);
        recentExportPaths = Preferences.decodePaths(recentPathsPref);
        overwrite = store.getBoolean(Preferences.DRAWING_TEMPLATE_EXPORT_OVERWRITE);

        return true;
    }

    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.RECENT_DRAWING_TEMPLATE_EXPORT_LOCATIONS, Preferences.encodePaths(recentExportPaths));
        store.setValue(Preferences.DRAWING_TEMPLATE_EXPORT_OVERWRITE, exportModel.overwrite);

        if (store.needsSaving())
            store.save();
    }

    public DrawingTemplateExportWizard() {
        setWindowTitle("Export Diagram Template");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences();

        ISessionContext ctx = SimanticsUI.getSessionContext();
        if (ctx == null)
            return;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return;

        exportModel = new ExportPlan(ctx, recentExportPaths);
        exportModel.project = project;
        exportModel.selection = selection;
        exportModel.overwrite = overwrite;
    }

    @Override
    public void addPages() {
        super.addPages();
        if (exportModel != null) {
            addPage(new DrawingTemplateExportPage(exportModel));
        } else {
            addPage(new NoProjectPage("Export Diagram Template"));
        }
    }

    @Override
    public boolean performFinish() {
        try {
            recentExportPaths.addFirst(exportModel.exportLocation.getAbsolutePath());
            Preferences.removeDuplicates(recentExportPaths);
            if (recentExportPaths.size() > MAX_RECENT_EXPORT_PATHS)
                recentExportPaths.pollLast();

            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        final File output = exportModel.exportLocation;
        if (output.exists()) {
            if (!exportModel.overwrite) {
                boolean ok = MessageDialog.openConfirm(getShell(), "Overwrite", "A file by the name " + output.getAbsolutePath() + " already exists.\n\nDo you want to overwrite?");
                if (!ok) {
                    return false;
                }
            }
            if (!output.delete()) {
                MessageDialog.openError(getShell(), "Delete Problem", "Could not overwrite previously existing file " + output);
                return false;
            }
        }

        try {
            getContainer().run(true, true, new DrawingTemplateExporter(exportModel));
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            WizardPage cp = (WizardPage) getContainer().getCurrentPage();
            if (t instanceof IOException) {
                ErrorLogger.defaultLogError("An I/O problem occurred while exporting diagram template. See exception for details.", t);
                cp.setErrorMessage("An I/O problem occurred while exporting diagram template.\n\nMessage: " + e.getMessage());
            } else {
                ErrorLogger.defaultLogError("Unexpected exception while exporting diagram template. See exception for details.", t);
                cp.setErrorMessage("Unexpected exception while exporting diagram template. See error log for details.\n\nMessage: " + e.getMessage());
            }
            return false;
        } catch (InterruptedException e) {
            ExceptionUtils.logAndShowError(e);
            return false;
        }

        return true;
    }

}
