/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.template2d.ui.diagram.adapter;


import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.geom.Line2D;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.util.Bean;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.diagram.G2DUtils;
import org.simantics.diagram.function.All;
import org.simantics.diagram.profile.StyleBase;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.g2d.scenegraph.SceneGraphConstants;
import org.simantics.modeling.template2d.ontology.Template2dResource;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.nodes.ShapeNode;
import org.simantics.scenegraph.g2d.nodes.spatial.RTreeNode;
import org.simantics.scenegraph.profile.DataNodeMap;
import org.simantics.scenegraph.profile.EvaluationContext;
import org.simantics.scenegraph.profile.common.ProfileVariables;
import org.simantics.utils.page.PageOrientation;

public class DrawingBorderStyle extends StyleBase<DrawingBorderStyle.StyleInfo> {
	private static final String BORDER_NODE = "template>border";
	
	private static int TOP_INDEX = 0;
	private static int RIGHT_INDEX = 1;
	private static int BOTTOM_INDEX = 2;
	private static int LEFT_INDEX = 3;

	private static int HEIGHT_INDEX = 1;
	private static int WIDTH_INDEX = 0;

	public static class StyleInfo extends Bean {
		@Optional
		public float[] pageSize = null;
		@Optional
		public float[] margin = null;
		@Optional
		public float[] borderSize = null;
		@Optional
		public float[] padding = null;
		@Optional
		public PageOrientation orientation=null;
		@Optional
		public float rgba[] = null;
		@Optional
		public Resource template = null;
	};


	@Override
	public StyleInfo calculateStyle(ReadGraph graph, Resource runtimeDiagram,
			Resource entry, Resource groupItem, Variable activeComposite)
			throws DatabaseException {
//        DiagramResource DIA = DiagramResource.getInstance(graph);
//        ModelingResources MOD = ModelingResources.getInstance(graph);
//        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        Template2dResource TEMPLATE2D = Template2dResource.getInstance(graph);

//        Layer0 L0 = Layer0.getInstance(graph);
        G2DResource G2D = G2DResource.getInstance(graph);
        
        StyleInfo info = new StyleInfo();
        info.template = All.getTemplate(graph, runtimeDiagram);
        
        //info.init();
        
        if (info.template == null)
        	return null;
        
        info.margin = graph.getPossibleRelatedValue(info.template, TEMPLATE2D.HasMargin, Bindings.FLOAT_ARRAY);
        if (info.margin == null)
        	return null;
        	
        info.padding = graph.getPossibleRelatedValue(info.template, TEMPLATE2D.HasPadding, Bindings.FLOAT_ARRAY);

        Resource orientation = graph.getPossibleObject(info.template, TEMPLATE2D.HasPageOrientation);
        info.orientation = PageOrientation.Portrait;
        if (orientation != null && orientation.equals(TEMPLATE2D.PageOrientation_Landscape))
        	info.orientation = PageOrientation.Landscape;
        
        Resource page = graph.getPossibleObject(info.template, TEMPLATE2D.HasPage);
        if (page == null)
        	return null;
        
        Resource border = graph.getPossibleObject(info.template, TEMPLATE2D.HasBorder);
        if (border == null)
        	return null;

        info.pageSize = graph.getPossibleRelatedValue(page, TEMPLATE2D.HasSize, Bindings.FLOAT_ARRAY);
        if (info.pageSize == null)
        	return null;
        
        Resource rColor = graph.getPossibleObject(border, G2D.HasColor);
        Color color = null; 
        if (rColor != null)
        	color = (Color) G2DUtils.getObject(graph, rColor);
        if (color == null)
        	color = new Color(0, 0, 0);
        
        info.rgba = new float[4];
        color.getRGBComponents(info.rgba);
       	
        info.borderSize = graph.getPossibleRelatedValue(border, TEMPLATE2D.HasSize, Bindings.FLOAT_ARRAY);
        if (info.borderSize == null)
        	return null;
        
        return info;
	}
	
	@Override
	public void applyStyleForItem(EvaluationContext evaluationContext, DataNodeMap map, Object item, StyleInfo info) {
		//super.applyStyleForItem(evaluationContext, map, item, value);
//		final INode node = map.getNode(item);
//		if (node == null) {
//			evaluationContext.update();
//			// TODO: continue or return?
//			return;
//		}
		if (info == null)
			return;
		
		float[] pageSize = info.pageSize;
		float[] margin = extract(info.margin);
		float[] borderSize = extract(info.borderSize);
		//float[] padding = extract(info.padding);
		
		if(pageSize == null || pageSize.length != 2 || margin == null || borderSize == null)
			return;
		
	    if (info.orientation == PageOrientation.Portrait && pageSize[WIDTH_INDEX] > pageSize[HEIGHT_INDEX]){
	    	float tmp = pageSize[HEIGHT_INDEX];
	    	pageSize[HEIGHT_INDEX] = pageSize[WIDTH_INDEX];
	    	pageSize[WIDTH_INDEX] = tmp;
		}
	    if (info.orientation == PageOrientation.Landscape && pageSize[WIDTH_INDEX] < pageSize[HEIGHT_INDEX]){
	    	float tmp = pageSize[HEIGHT_INDEX];
	    	pageSize[HEIGHT_INDEX] = pageSize[WIDTH_INDEX];
	    	pageSize[WIDTH_INDEX] = tmp;
		}

	    Line2D[] borders = new Line2D[4];
		borders[TOP_INDEX] = new Line2D.Float(margin[LEFT_INDEX], margin[TOP_INDEX]+(borderSize[TOP_INDEX]/2.0f), pageSize[WIDTH_INDEX]-margin[RIGHT_INDEX], margin[TOP_INDEX]+(borderSize[TOP_INDEX]/2.0f));
		borders[BOTTOM_INDEX] = new Line2D.Float(margin[LEFT_INDEX], pageSize[HEIGHT_INDEX]-margin[BOTTOM_INDEX]-(borderSize[BOTTOM_INDEX]/2.0f), pageSize[WIDTH_INDEX]-margin[RIGHT_INDEX], pageSize[HEIGHT_INDEX]-margin[BOTTOM_INDEX]-(borderSize[BOTTOM_INDEX]/2.0f));
		borders[LEFT_INDEX] = new Line2D.Float(margin[LEFT_INDEX]+(borderSize[LEFT_INDEX]/2.0f), margin[TOP_INDEX]+(borderSize[TOP_INDEX]/2.0f), margin[LEFT_INDEX]+(borderSize[LEFT_INDEX]/2.0f), pageSize[HEIGHT_INDEX]-margin[BOTTOM_INDEX]-(borderSize[BOTTOM_INDEX]/2.0f));
		borders[RIGHT_INDEX] = new Line2D.Float(pageSize[WIDTH_INDEX]-margin[RIGHT_INDEX]-(borderSize[RIGHT_INDEX]/2.0f), margin[TOP_INDEX]+(borderSize[TOP_INDEX]/2.0f), pageSize[WIDTH_INDEX]-margin[RIGHT_INDEX]-(borderSize[RIGHT_INDEX]/2.0f), pageSize[HEIGHT_INDEX]-margin[BOTTOM_INDEX]-(borderSize[BOTTOM_INDEX]/2.0f));
		

		G2DSceneGraph sg = evaluationContext.getSceneGraph();
		if (sg == null)
			return;
		//NodeUtil.printSceneGraph(sg);
		G2DParentNode nav = (G2DParentNode) sg.getNode(SceneGraphConstants.NAVIGATION_NODE_NAME);
		//NodeUtil.lookup(sg, SceneGraphConstants.NAVIGATION_NODE_NAME, G2DParentNode.class);
		if (nav == null)
			return;

	    cleanupStyleForItem(evaluationContext, map, item);

	    RTreeNode pageTemplate = ProfileVariables.claimChild(evaluationContext.getSceneGraph(),
                SceneGraphConstants.NAVIGATION_NODE_NAME, BORDER_NODE, RTreeNode.class, //SingleElementNode.class,
                evaluationContext);
        if (pageTemplate == null)
            return;

	    pageTemplate.setZIndex(1);
	   // pageTemplate.setVisible(true);

//	    Path2D path = new Path2D.Float();
//	    
//	    path.append(line, false);
	    
//		BasicStroke[] strokes = new BasicStroke[4];
	    //GeneralPath p = new GeneralPath();
	    
		for (int i=0;i<4;i++){
//			if (i==1 || i == 3)
//				continue;
			Line2D line = borders[i];
			//Rectangle2D rect = new Rectangle2D.Float(0,0,100,100);
			if (borderSize[i] < 0)
				continue;
			BasicStroke stroke = new BasicStroke(borderSize[i], BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
		    ShapeNode shape = pageTemplate.addNode("border_" + i, ShapeNode.class); // SingleElementNode
	        shape.setShape(line);
	        shape.setScaleStroke(false);
	        shape.setStroke(stroke);
	        shape.setFill(false);
	        shape.setColor(new Color(info.rgba[0], info.rgba[1], info.rgba[2], info.rgba[3]));
	        if (i == TOP_INDEX || i == BOTTOM_INDEX)
	        	shape.setZIndex(10);

//	        System.err.println(borderSize[i]);
//	        System.err.print(borders[i].getX1());
//	        System.err.print("," + borders[i].getY1());
//	        System.err.print("," + borders[i].getX2());
//	        System.err.println("," + borders[i].getY2());
//	        if (i == 2){
//				BasicStroke stroke2 = new BasicStroke(0.1f);
//			    ShapeNode shape2 = pageTemplate.addNode("border_" + 4, ShapeNode.class); // SingleElementNode
//		        shape2.setShape(line);
//		        shape2.setScaleStroke(true);
//		        shape2.setStroke(stroke2);
//		        shape2.setFill(false);
//		        shape2.setZIndex(10);
//		        shape2.setColor(new Color(0, 0, 0));
//	        }
		}
		
//		if (DebugPolicy.DEBUG_PROFILE_STYLE_APPLICATION)
//			System.out.println(StyleBase.this + ": applying style for item " + item + " and element " + node + " with result " + value);
	}

    protected void cleanupStyleForItem(EvaluationContext evaluationContext, DataNodeMap map, Object item) {
        ProfileVariables.denyChild(evaluationContext.getSceneGraph(), SceneGraphConstants.NAVIGATION_NODE_NAME,
                BORDER_NODE);
       // System.err.println("template/border removed");
    }

    @Override
    public String toString() {
        return "Border style";
    }

    private float[] extract(float[] settings){
    	float[] ret = new float[]{0.0f,0.0f,0.0f,0.0f};
		if (settings == null || settings.length == 0 || settings.length > 4){
			return null;
		}
		if (settings.length == 1){
			ret[TOP_INDEX] = settings[0];
			ret[BOTTOM_INDEX] = settings[0];
			ret[LEFT_INDEX] = settings[0];
			ret[RIGHT_INDEX] = settings[0];
		}
		if (settings.length == 2){
			ret[TOP_INDEX] = settings[0];
			ret[BOTTOM_INDEX] = settings[0];
			ret[LEFT_INDEX] = settings[1];
			ret[RIGHT_INDEX] = settings[1];
		}
		if (settings.length == 3){
			ret[TOP_INDEX] = settings[0];
			ret[BOTTOM_INDEX] = settings[2];
			ret[LEFT_INDEX] = settings[1];
			ret[RIGHT_INDEX] = settings[1];
		}
		if (settings.length == 4){
			ret[TOP_INDEX] = settings[0];
			ret[RIGHT_INDEX] = settings[1];
			ret[BOTTOM_INDEX] = settings[2];
			ret[LEFT_INDEX] = settings[3];
		}
    	return ret;
    }
}
