package org.simantics.db.layer0.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ResourceMap;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.procedure.AsyncContextMultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.service.DirectQuerySupport;
import org.simantics.layer0.Layer0;

class ConsistsOfProcess {

	final List<InternalEntry> result;
	final AsyncContextMultiProcedure<InternalEntry, Resource> structure;
	final AsyncContextMultiProcedure<InternalEntry, Resource> names;

    public static List<InternalEntry> walk(ReadGraph graph, ResourceMap<ExtentStatus> status, Collection<Resource> resources, Set<Resource> exclusions, boolean ignoreVirtual) throws DatabaseException {
    	ConsistsOfProcess process = new ConsistsOfProcess(graph, status, resources, exclusions, ignoreVirtual);
    	return process.result;
    }
    
    static class InternalEntry {
    	public InternalEntry parent;
    	public Resource resource;
    	public String name;
    	InternalEntry(InternalEntry parent, Resource resource, String name) {
    		this.parent = parent;
    		this.resource = resource;
    		this.name = name;
    	}
    }
	
    private ConsistsOfProcess(ReadGraph graph, ResourceMap<ExtentStatus> status, final Collection<Resource> resources, final Set<Resource> exclusions, final boolean ignoreVirtual) throws DatabaseException {

		final Layer0 L0 = Layer0.getInstance(graph);
		final DirectQuerySupport dqs = graph.getService(DirectQuerySupport.class);
		
		result = new ArrayList<InternalEntry>();
		
		names = dqs.compileForEachObject(graph, L0.HasName, new AsyncContextMultiProcedure<InternalEntry, Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, InternalEntry entry, Resource nameResource) {
		
				if(status != null)
					status.put(nameResource, ExtentStatus.EXCLUDED);
				
				graph.forPossibleValue(nameResource, new Procedure<String>() {

					@Override
					public void execute(String result) {
						entry.name = result;
					}

					@Override
					public void exception(Throwable t) {
						Logger.defaultLogError(t);
					}
					
				});
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				Logger.defaultLogError(throwable);
			}

			@Override
			public void finished(AsyncReadGraph graph) {
			}

		});
		
		structure = dqs.compileForEachObject(graph, L0.ConsistsOf, new AsyncContextMultiProcedure<InternalEntry, Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, InternalEntry parent, Resource child) {
				
				if(exclusions.contains(child)) return;
				
				if(!ignoreVirtual || child.isPersistent()) {
					InternalEntry entry = new InternalEntry(parent, child, null);
					if(result.add(entry)) {
						dqs.forEachObjectCompiled(graph, child, entry, structure);
						dqs.forEachObjectCompiled(graph, child, entry, names);
					}
				}
				
			}

			@Override
			public void finished(AsyncReadGraph graph) {
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				Logger.defaultLogError(throwable);
			}

		});
		
		graph.syncRequest(new ReadRequest() {

			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				for(Resource r  : resources) {
					InternalEntry root = new InternalEntry(null, r, null);
					dqs.forEachObjectCompiled(graph, r, root, structure);
				}
			}
			
		});
		
	}
	
    
}
