package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class EApplyType extends Expression {    
    Expression expression;
    Type parameter;
    
    public EApplyType(Expression expression, Type parameter) {
        this.expression = expression;
        this.parameter = parameter;
    }
    
    public static Expression create(Expression expression, Type ... parameters) {
        for(Type parameter : parameters)
            expression = new EApplyType(expression, parameter);
        return expression;
    }

    public EApplyType(long loc, Expression expression, Type parameter) {
        super(loc);
        if(parameter == null)
            throw new NullPointerException();
        this.expression = expression;
        this.parameter = parameter;
    }
    
    public Expression getExpression() {
        return expression;
    }
    
    public Type getParameter() {
        return parameter;
    }

	public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        expression.collectRefs(allRefs, refs);
    }
	
	public void collectVars(TObjectIntHashMap<Variable> allVars, TIntHashSet vars) {
	    expression.collectVars(allVars, vars);
	}

	@Override
	protected void updateType() throws MatchException {
	    setType(Types.instantiate(expression.getType(), parameter));
	}
	
	@Override
	public IVal toVal(Environment env, CodeWriter w) {
        IVal val = expression.toVal(env, w);
        return val.createSpecialization(parameter);
    }

    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
        expression.collectFreeVariables(vars);
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        expression = expression.simplify(context);
        if(expression instanceof EConstant) {
            ((EConstant)expression).addTypeParameters(parameter);
            return expression;
        }
        return this;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        expression = expression.resolve(context);
        return this;
    }
    
    @Override
    public void removeFreeVariables(THashSet<Variable> vars) {
        expression.removeFreeVariables(vars);
    }

    @Override
    public Expression replace(ReplaceContext context) {
        return new EApplyType(expression.replace(context), 
                parameter.replace(context.tvarMap));
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext target) {
        return expression.toIExpression(target);
    }
    
    @Override
    public Expression inferType(TypingContext context) {
        throw new InternalCompilerError("Should not type check " + getClass().getSimpleName() + ".");
    }

    @Override
    public Expression decorate(ExpressionDecorator decorator) {
        expression = expression.decorate(decorator);
        return decorator.decorate(this);
    }
    
    @Override
    public boolean isEffectful() {
    	return expression.isEffectful();
    }

    @Override
    public void collectEffects(THashSet<Type> effects) {
        expression.collectEffects(effects);
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            expression.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public void forVariables(VariableProcedure procedure) {
        expression.forVariables(procedure);
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }

}
