package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.internal.parsing.Symbol;
import org.simantics.scl.compiler.types.Type;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class Case extends Symbol {
    public Expression[] patterns;
    public Expression value;
    
    long lhs;
    
    public Case(Expression[] patterns, Expression value) {
        this.patterns = patterns;
        this.value = value;
    }
    
    public Case(Expression pattern, Expression value) {
        this(new Expression[] {pattern}, value);
    }

    public void setLhs(long lhs) {
        this.lhs = lhs;
    }
    
    public long getLhs() {
        return lhs;
    }

	public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        value.collectRefs(allRefs, refs);
    }

    public void collectVars(TObjectIntHashMap<Variable> allVars,
            TIntHashSet vars) {
        value.collectVars(allVars, vars);
    }

    public void collectFreeVariables(THashSet<Variable> vars) {
        value.collectFreeVariables(vars);
        for(int i=patterns.length-1;i>=0;--i)
            patterns[i].removeFreeVariables(vars);
    }

    public void resolve(TranslationContext context) {
        context.pushFrame();
        for(int i=0;i<patterns.length;++i)
            patterns[i] = patterns[i].resolveAsPattern(context);
        value = value.resolve(context);
        context.popFrame();
    }

    public void simplify(SimplificationContext context) {
        for(int i=0;i<patterns.length;++i)
            patterns[i] = patterns[i].simplify(context);
        value = value.simplify(context);
    }

    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            for(Expression pattern : patterns)
                pattern.setLocationDeep(loc);
            value.setLocationDeep(loc);
        }
    }

    public Case replace(ReplaceContext context) {
        Expression[] newPatterns = new Expression[patterns.length];        
        for(int i=0;i<patterns.length;++i)
            newPatterns[i] = patterns[i].replaceInPattern(context);
        Expression newValue = value.replace(context);
        Case result = new Case(newPatterns, newValue);
        result.setLhs(lhs);
        return result;
    }

    public Expression[] getPatterns() {
        return patterns;
    }

    public void checkType(TypingContext context, Type[] parameterTypes,
            Type requiredType) {
        if(patterns.length != parameterTypes.length) {
            context.getErrorLog().log(location, "This case has different arity ("+patterns.length+
                    ") than than the first case (+"+parameterTypes.length+"+).");
            return;
        }
        for(int i=0;i<patterns.length;++i)
            patterns[i] = patterns[i].checkTypeAsPattern(context, parameterTypes[i]);
        value = value.checkType(context, requiredType);
    }
    
    public void checkIgnoredType(TypingContext context, Type[] parameterTypes) {
        if(patterns.length != parameterTypes.length) {
            context.getErrorLog().log(location, "This case has different arity ("+patterns.length+
                    ") than than the first case (+"+parameterTypes.length+"+).");
            return;
        }
        for(int i=0;i<patterns.length;++i)
            patterns[i] = patterns[i].checkTypeAsPattern(context, parameterTypes[i]);
        value = value.checkIgnoredType(context);
    }
        
    public void decorate(ExpressionDecorator decorator) {
        for(int i=0;i<patterns.length;++i)
            patterns[i] = patterns[i].decorate(decorator);
        value = value.decorate(decorator);
    }

    public void forVariables(VariableProcedure procedure) {
        value.forVariables(procedure);
    }
}
