package org.simantics.scl.compiler.elaboration.chr.planning.items;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.elaboration.chr.CHRLiteral;
import org.simantics.scl.compiler.elaboration.chr.CHRRelation;
import org.simantics.scl.compiler.elaboration.chr.plan.CheckOp;
import org.simantics.scl.compiler.elaboration.chr.plan.IterateConstraintOp;
import org.simantics.scl.compiler.elaboration.chr.plan.IterateRelationOp;
import org.simantics.scl.compiler.elaboration.chr.planning.PrePlanItem;
import org.simantics.scl.compiler.elaboration.chr.planning.QueryPlanningContext;
import org.simantics.scl.compiler.elaboration.chr.relations.CHRConstraint;
import org.simantics.scl.compiler.elaboration.chr.relations.ExternalCHRRelation;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.expressions.Variable;
import org.simantics.scl.compiler.elaboration.java.Builtins;
import org.simantics.scl.compiler.elaboration.relations.SCLRelation;

import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class GenericPrePlanItem extends PrePlanItem {
    public CHRLiteral literal;
    public CHRRelation relation;
    public Expression[] expressions;
    public TIntHashSet[] variableSets;
    TIntHashSet allVars;

    public GenericPrePlanItem(CHRLiteral literal, CHRRelation relation, Expression[] expressions,
            TIntHashSet[] variableSets, int secondaryPriority) {
        super(secondaryPriority);
        this.literal = literal;
        this.relation = relation;
        this.expressions = expressions;
        this.variableSets = variableSets;
        allVars = new TIntHashSet();
        for(TIntHashSet variableSet : variableSets)
            allVars.addAll(variableSet);
        updatePrimaryPriority();
    }

    private void updatePrimaryPriority() {
        int boundCount = 0;
        int boundMask = 0;
        for(int i=0;i<variableSets.length;++i)
            if(variableSets[i].isEmpty()) {
                ++boundCount;
                boundMask |= 1 << i;
            }
        if(boundCount == variableSets.length)
            primaryPriority = 0;
        else {
            if(relation instanceof ExternalCHRRelation) {
                SCLRelation sclRelation = ((ExternalCHRRelation)relation).relation;
                double selectivity = sclRelation.getSelectivity(boundMask);
                if(selectivity == Double.POSITIVE_INFINITY)
                    primaryPriority = Double.POSITIVE_INFINITY;
                else if(selectivity < 1.0)
                    primaryPriority = 0.0;
                else if(selectivity == 1.0)
                    primaryPriority = 1.0;
                else
                    primaryPriority = 3.0 - ((double)boundCount) / variableSets.length;
            }
            else
                primaryPriority = 3.0 - ((double)boundCount) / variableSets.length;
        }
    }

    @Override
    public void initializeListeners(QueryPlanningContext context) {
        context.listen(allVars, this);
    }

    @Override
    public void variableSolved(QueryPlanningContext context, int variableId) {
        for(TIntHashSet variableSet : variableSets)
            variableSet.remove(variableId);
        allVars.remove(variableId);
        updatePrimaryPriority();
        context.priorityQueue.adjust(this);
    }
    
    @Override
    public void generate(QueryPlanningContext context) {
        int boundMask = 0;
        Expression[] boundExpressions = new Expression[expressions.length];
        Variable[] freeVariables = new Variable[expressions.length];
        int freeVariableCount = 0;
        for(int i=0;i<expressions.length;++i)
            if(variableSets[i].isEmpty()) {
                boundExpressions[i] = expressions[i];
                boundMask |= 1 << i;
            }
            else {
                freeVariables[i] = ((EVariable)expressions[i]).getVariable();
                ++freeVariableCount;
            }
        if(relation instanceof CHRConstraint)
            context.addPlanOp(new IterateConstraintOp(location, (CHRConstraint)relation, freeVariables, boundExpressions, boundMask,
                    killAfterMatch(), literal.passive));
        else if(relation instanceof ExternalCHRRelation)
            context.addPlanOp(new IterateRelationOp(location, ((ExternalCHRRelation)relation).relation,
                    freeVariables, boundExpressions, literal.typeConstraintEvidenceParameters, boundMask));
        else
            throw new InternalCompilerError();
        if(freeVariableCount > 1) {
            THashSet<Variable> usedVariables = new THashSet<Variable>(freeVariableCount);
            for(int i=0;i<freeVariables.length;++i) {
                Variable variable = freeVariables[i];
                if(variable == null)
                    continue;
                if(!usedVariables.add(variable)) {
                    Variable auxiliary = new Variable(variable.getName(), variable.getType());
                    freeVariables[i] = auxiliary;
                    context.addPlanOp(new CheckOp(location, new EApply(location, new EConstant(Builtins.EQUALS, variable.getType()), new EVariable(auxiliary), new EVariable(variable))));
                }
            }
        }
        context.bind(allVars);
    }

    private boolean killAfterMatch() {
        return literal.killAfterMatch && relation instanceof CHRConstraint;
    }
}
