package org.simantics.scl.compiler.constants.generic;

import org.cojen.classfile.TypeDesc;
import org.simantics.scl.compiler.constants.FunctionValue;
import org.simantics.scl.compiler.internal.codegen.references.Val;
import org.simantics.scl.compiler.internal.codegen.types.JavaReferenceValidator;
import org.simantics.scl.compiler.internal.codegen.utils.MethodBuilder;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Type;

/**
 * This class represents a call to a Java method as an SCL function value.
 * It is instantiated by the {@link org.simantics.scl.compiler.compilation.Elaboration.matchType(MethodRef, Type)} method. 
 */
public class CallJava extends FunctionValue {
    StackItem[] stackItems;
    MethodRef methodRef;
    OutputFilter filter;
    
    public CallJava(TVar[] typeParameters, Type effect, Type returnType,
            Type[] parameterTypes, StackItem[] stackItems, MethodRef methodRef,
            OutputFilter filter) {
        super(typeParameters, effect, returnType, parameterTypes);
        if(stackItems == null) {
            stackItems = new StackItem[parameterTypes.length];
            for(int i=0;i<parameterTypes.length;++i)
                stackItems[i] = new ParameterStackItem(i, parameterTypes[i]);
        }
        this.stackItems = stackItems;
        this.methodRef = methodRef;
        this.filter = filter;
    }

    @Override
    public Type applyExact(MethodBuilder mb, Val[] parameters) {
        methodRef.invoke(mb, stackItems, parameters);
        if(filter != null)
            filter.filter(mb);
        return getReturnType();
    }
    
    public MethodRef getMethodRef() {
        return methodRef;
    }
    
    @Override
    public String toString() {
        return methodRef.getName();
    }

    public final static int INCOMPARABLE = -2;
    public final static int LESS = -1;
    public final static int EQUAL = 0;
    public final static int GREATER = 1;
    
    public int compareTo(JavaReferenceValidator validator, CallJava other) {
        MethodRef m1 = methodRef;
        MethodRef m2 = other.methodRef;
        TypeDesc[] ps1 = m1.getParameterTypes();
        TypeDesc[] ps2 = m2.getParameterTypes();
        if(ps1.length != ps2.length)
            return INCOMPARABLE;

        boolean lessOrEqual = true;
        boolean greaterOrEqual = true;
        for(int i=0;i<ps1.length;++i) {
            if(ps1[i].equals(ps2[i]))
                continue;
            if(!validator.isAssignableFrom(ps1[i], ps2[i])) {
                lessOrEqual = false;
            }
            if(!validator.isAssignableFrom(ps2[i], ps1[i])) {
                greaterOrEqual = false;
            }
        }
        if(lessOrEqual) {
            if(greaterOrEqual)
                return EQUAL;
            else
                return LESS;
        }
        else {
            if(greaterOrEqual)
                return GREATER;
            else
                return INCOMPARABLE;
        }
    }
    
    @Override
    public void prepare(MethodBuilder mb) {
        for(StackItem item : stackItems)
            item.prepare(mb);
    }
}
