/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.sharedontology.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.util.DraftStatusBean;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.ModelingUtils.LibraryInfo;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Antti Villberg
 */
public class ModelExportPage extends WizardPage {

    ExportPlan          exportModel;
    Composite           draft;
    CCombo              model;
    CCombo              exportLocation;

    List<LibraryInfo> models = Collections.emptyList();
    private Button      overwrite;

    protected ModelExportPage(ExportPlan model) {
        super("Export Model", "Define Export Location", null);
        this.exportModel = model;
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        draft = new Composite(container, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(draft);
        draft.setBackground(draft.getDisplay().getSystemColor(SWT.COLOR_RED));
        GridLayoutFactory.swtDefaults().spacing(5, 5).applyTo(draft);

        Composite draft2 = new Composite(draft, SWT.NONE);
        GridLayoutFactory.swtDefaults().applyTo(draft2);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(draft2);
        Label draftText = new Label(draft2, SWT.WRAP);
        draftText.setText("Some dependencies of the selected model are not published. The model can only be saved with draft status.");

        new Label(container, SWT.NONE).setText("Exported &model:");
        model = new CCombo(container, SWT.BORDER);
        {
            model.setEditable(false);
            model.setText("");
            model.setToolTipText("Selects the model to export.");
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(model);
            model.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    int selected = model.getSelectionIndex();
                    if (selected != -1) {
                        exportModel.model = (LibraryInfo) model.getData(String.valueOf(selected)); 
                        validatePage();
                    }
                }
            });
        }

        new Label(container, SWT.NONE).setText("&Target file:");
        exportLocation = new CCombo(container, SWT.BORDER);
        {
            exportLocation.setText("");
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(exportLocation);
            exportLocation.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText("Browse...");
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
                    dialog.setText("Choose Export Target File");
                    String loc = exportLocation.getText();
                    dialog.setFilterPath(loc);
                    dialog.setFilterExtensions(new String[] { "*.tg" });
                    dialog.setFilterNames(new String[] { "Model (*.tg)" });
                    dialog.setOverwrite(false);
                    String file = dialog.open();
                    if (file == null)
                        return;
                    exportLocation.setText(file);
                    validatePage();
                }
            });
        }

        Label horizRule = new Label(container, SWT.BORDER);
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 0).grab(true, false).span(3, 1).applyTo(horizRule);

        overwrite = new Button(container, SWT.CHECK);
        overwrite.setText("&Overwrite existing files without warning");
        overwrite.setSelection(exportModel.overwrite);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(overwrite);
        overwrite.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                validatePage();
            }
        });

        try {
            initializeData();
        } catch (DatabaseException e) {
            e.printStackTrace();
        }

        setControl(container);
        validatePage();
    }

    private void initializeData() throws DatabaseException {

        List<Resource> selected = ISelectionUtils.getPossibleKeys(exportModel.selection, SelectionHints.KEY_MAIN, Resource.class);

        models = exportModel.sessionContext.getSession().syncRequest(new UniqueRead<List<LibraryInfo>>() {
            @Override
            public List<LibraryInfo> perform(ReadGraph graph) throws DatabaseException {
                List<LibraryInfo> result = new ArrayList<>();
                Layer0 L0 = Layer0.getInstance(graph);
                for (Resource r : selected) {
                    String name = graph.getRelatedValue(r, L0.HasName, Bindings.STRING);
                    DraftStatusBean draft = ModelingUtils.getDependencyDraftStatus(graph, r);
                    result.add(new LibraryInfo(name, r, draft));
                }
                Collections.sort(result, new Comparator<LibraryInfo>() {
                    @Override
                    public int compare(LibraryInfo o1, LibraryInfo o2) {
                        return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(o1.library.getName(), o2.library.getName());
                    }
                });
                return result;
            }
        });

        if (!models.isEmpty())
            exportModel.model = models.get(0);

        // Populate combo boxes
        int i = 0;
        for (LibraryInfo m : models) {
            model.add(m.library.getName());
            model.setData(String.valueOf(i), m);
            if (m.equals(exportModel.model))
                model.select(i);
            ++i;
        }

        for (String path : exportModel.recentLocations) {
            exportLocation.add(path);
        }
        if (exportLocation.getItemCount() > 0)
            exportLocation.select(0);
    }

    void validatePage() {
        if (exportModel.model == null) {
            setMessage("Select library to export from.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }

        boolean draftVisible = draft.getVisible();
        boolean hasDraftDependencies = exportModel.model.draft != null;
        if (draftVisible != hasDraftDependencies) {
            ((GridData) draft.getLayoutData()).exclude = !hasDraftDependencies;
            draft.setVisible(hasDraftDependencies);
            draft.getParent().getParent().layout(true, true);
        }

        String exportLoc = exportLocation.getText();
        if (exportLoc.isEmpty()) {
            setMessage("Select target file.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        File file = new File(exportLoc);
        if (file.isDirectory()) {
            setErrorMessage("The target is a directory.");
            setPageComplete(false);
            return;
        }
        File parent = file.getParentFile();
        if (parent == null || !parent.isDirectory()) {
            setErrorMessage("The target directory does not exist.");
            setPageComplete(false);
            return;
        }
        exportModel.exportLocation = file;
        exportModel.overwrite = overwrite.getSelection();

        setErrorMessage(null);
        setMessage("Export selected model to " + exportModel.exportLocation + ".");
        setPageComplete(true);
    }

}
