/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property;

import java.util.List;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.part.IPage;
import org.eclipse.ui.part.PageBookView;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.common.property.IProperty;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.management.ISessionContext;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.IPropertyPage;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ISelectionUtils;


/**
 * @author Tuukka Lehtonen
 */
public class RestoreDefaultValueHandler extends AbstractHandler {

    IPage getCurrentPage(IWorkbenchPart part) {
        if (part instanceof PageBookView)
            return ((PageBookView) part).getCurrentPage();
        if (part instanceof org.simantics.browsing.ui.platform.PageBookView)
            return ((org.simantics.browsing.ui.platform.PageBookView) part).getCurrentPage();
        return null;
    }

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        // Cannot use this, since the property view does not contribute a selection back to the workbench.
        //HandlerUtil.getCurrentSelection(event);

        ISessionContext sc = SimanticsUI.getSessionContext();
        if (sc == null)
            return null;

        IWorkbenchPart part = HandlerUtil.getActivePartChecked(event);
        IPage page = getCurrentPage(part);
        if (page == null || !(page instanceof IPropertyPage))
            return null;
        IPropertyPage propPage = (IPropertyPage) page;
        ISelection sel = propPage.getSelection();
        if (sel == null)
            return null;

        final List<Variable> vars = ISelectionUtils.filterSelection(sel, Variable.class);
        if (!vars.isEmpty()) {
            resetVariableProperties(sc, vars);
        } else {
            final List<IProperty> props = ISelectionUtils.filterSelection(sel, IProperty.class);
            if (!props.isEmpty())
                resetLegacyProperties(sc, props);
        }

        return null;
    }


    private void resetVariableProperties(ISessionContext sc, final List<Variable> vars) {
        // TODO: how to do this generically using only variables? How to remove a property value to let it be asserted again?
        sc.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                for (Variable v : vars) {
//                    Variable predicate = v.getPossiblePropertyValue(graph, Variables.PREDICATE);
//                    if (predicate == null)
//                        continue;
//                    Resource predicateResource = predicate.getPossiblePropertyValue(graph, Variables.REPRESENTS);
//                    if (predicateResource == null)
//                        continue;
                	Resource predicateResource = v.getPossiblePredicateResource(graph);
                	if (predicateResource == null)
                		continue;
                    Variable parent = v.getParent(graph);
                    if (parent == null)
                        continue;
                    Resource parentResource = parent.getPossibleRepresents(graph);
                    if (parentResource == null)
                        continue;

//                    System.out.println(parent.getURI(graph));
//                    System.out.println(predicate.getURI(graph));
//                    System.out.println(v.getURI(graph));

                    graph.denyValue(parentResource, predicateResource);
                    cm.add("Restored default value for property " + NameUtils.getSafeName(graph, predicateResource));
                }
                graph.addMetadata(cm);
            }
        }, new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException parameter) {
                if (parameter != null)
                    ErrorLogger.defaultLogError("Failed to restore default property values, see exception for details.", parameter);
            }
        });
    }

    private boolean resetLegacyProperties(ISessionContext sc, final List<IProperty> props) {
        // Dry run to see if there's anything to remove before writing.
        boolean changes = false;
        for (IProperty cp : props) {
            if (IProperty.DIRECT.contains(cp.getType())) {
                changes = true;
                break;
            }
        }
        if (changes) {
            sc.getSession().asyncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    for (IProperty cp : props) {
                        if (IProperty.DIRECT.contains(cp.getType())) {
                            Resource[] data = cp.getData(Resource[].class);
                            graph.denyStatement(data[0], data[1], data[2]);
                        }
                    }
                }
            }, new Callback<DatabaseException>() {
                @Override
                public void run(DatabaseException parameter) {
                    if (parameter != null)
                        ErrorLogger.defaultLogError("Failed to restore default property values, see exception for details.", parameter);
                }
            });
        }
        return changes;
    }

}
