/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.ListenerSupport;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.IToolMode;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.DataElementMap;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Callback;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramEditorStates {

    public static EditorState toEditorState(ICanvasContext ctx) {
        return toEditorState(ctx, true, true, true);
    }

    public static EditorState toEditorState(ICanvasContext ctx, boolean saveTransform, boolean saveSelection, boolean saveToolMode) {
        EditorState state = new EditorState();

        if (saveTransform) {
            state.viewTransform = ctx.getHintStack().getHint( Hints.KEY_CANVAS_TRANSFORM );
            if (state.viewTransform != null && state.viewTransform.getDeterminant() == 0)
                state.viewTransform = null;
        }

        if (saveToolMode) {
            IToolMode toolMode = ctx.getHintStack().getHint( Hints.KEY_TOOL );
            if (toolMode != null)
                state.toolMode = toolMode.getId();
        }

        if (saveSelection) {
            Set<IElement> selection = Collections.emptySet();
            for (Selection s : ctx.getItemsByClass( Selection.class ))
                selection = s.getSelection(0);
            state.selection = toResources( selection );
        }

        return state;
    }

    public static void saveEditorState(String virtualGraph, Resource diagram, ICanvasContext ctx, ListenerSupport support) {
        saveEditorState( virtualGraph, diagram, toEditorState(ctx), support );
    }

    public static Set<Resource> toResources(Set<IElement> elements) {
        Set<Resource> result = new HashSet<Resource>();
        for (IElement e : elements) {
            Object obj = ElementUtils.getObject(e);
            if (obj instanceof Resource)
                result.add((Resource) obj);
        }
        return result;
    }

    public static Set<IElement> toElements(Set<Resource> selection, IDiagram diagram) {
        Set<IElement> result = new HashSet<IElement>();
        DataElementMap dem = diagram.getDiagramClass().getAtMostOneItemOfClass(DataElementMap.class);
        for (Resource selected : selection) {
            IElement e = dem.getElement(diagram, selected);
            if (e != null)
                result.add(e);
        }
        return result;
    }

    public static void saveEditorState(final String virtualGraph, final Resource diagram, final EditorState editorState, final ListenerSupport support) {
        Session session = Simantics.getSession();

        final VirtualGraph vg = virtualGraph == null ? null
                : session.getService(VirtualGraphSupport.class).getWorkspacePersistent(virtualGraph);

        session.asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                // Prevent the state writing from failing in RemoverUtil.remove.
                if (Layer0Utils.isContainerPublished(graph, diagram))
                    return;

                // Remove all previous editor states.
                DiagramResource DIA = DiagramResource.getInstance(graph);
                for (Resource state : graph.getObjects(diagram, DIA.HasEditorState))
                    RemoverUtil.remove(graph, state);

                graph.syncRequest(new WriteRequest(vg) {
                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        newEditorState(graph, diagram, editorState);
                    }
                });
            }
        }, new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException parameter) {
                if (parameter != null)
                    support.exception(parameter);
            }
        });
    }

    private static Resource newEditorState(WriteGraph graph, Resource stateOf, EditorState editorState) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        G2DResource G2D = G2DResource.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);

        Resource state = graph.newResource();
        graph.claim(state, L0.InstanceOf, null, DIA.EditorState);
        graph.claim(stateOf, DIA.HasEditorState, state);

        if (editorState.viewTransform != null) {
            double[] matrix = new double[6];
            editorState.viewTransform.getMatrix(matrix);
            graph.claimLiteral(state, DIA.EditorState_ViewTransform, G2D.Transform, matrix, Bindings.DOUBLE_ARRAY);
        }

        if (editorState.toolMode != null)
            graph.claimLiteral(state, DIA.EditorState_ToolMode, L0.String, editorState.toolMode, Bindings.STRING);

        for (Resource selected : editorState.selection)
            graph.claim(state, DIA.EditorState_Selection, null, selected);

        return state;
    }

    public static EditorState readEditorState(final Resource source) throws DatabaseException {
        return Simantics.getSession().syncRequest(new ResourceRead<EditorState>(source) {
            @Override
            public EditorState perform(ReadGraph graph) throws DatabaseException {
                EditorState state = new EditorState();
                DiagramResource DIA = DiagramResource.getInstance(graph);
                for (Resource editorState : graph.getObjects(source, DIA.HasEditorState)) {
                    state.viewTransform = DiagramGraphUtil.getAffineTransform(graph, editorState, DIA.EditorState_ViewTransform, false);
                    state.toolMode = graph.getPossibleRelatedValue(editorState, DIA.EditorState_ToolMode, Bindings.STRING);
                    state.selection = new HashSet<Resource>();
                    for (Resource selected : graph.getObjects(editorState, DIA.EditorState_Selection)) {
                        if (graph.isInstanceOf(selected, DIA.Element))
                            state.selection.add(selected);
                    }
                    break;
                }
                return state;
            }
        });
    }

}
