/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagram.style;

import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.diagram.profile.StyleBase;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.issues.Severity;
import org.simantics.issues.common.IssueResourcesContexts;
import org.simantics.issues.common.ListModelIssuesBySeverity;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.diagram.style.IssueDecorationStyle.IssueResult;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.nodes.Decoration;
import org.simantics.scenegraph.g2d.nodes.DecorationSVGNode;
import org.simantics.scenegraph.g2d.nodes.SVGNode;
import org.simantics.scenegraph.profile.EvaluationContext;
import org.simantics.scenegraph.profile.common.ProfileVariables;
import org.simantics.scenegraph.utils.NodeUtil;
import org.simantics.utils.datastructures.map.Tuple;


/**
 * @author Tuukka Lehtonen
 */
public class IssueDecorationStyle extends StyleBase<IssueResult> {

    private static final String DECORATION_NODE_NAME = "issueDecorations";

    private List<Resource> getContexts(ReadGraph graph, Resource element) throws DatabaseException {

        ModelingResources MOD = ModelingResources.getInstance(graph);
        List<Resource> result = new ArrayList<Resource>(3);
        result.add(element);
        Resource config = graph.getPossibleObject(element, MOD.ElementToComponent);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        config = graph.getPossibleObject(element, MOD.DiagramConnectionToConnection);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        // For diagram reference element support
        config = graph.getPossibleObject(element, MOD.HasParentComponent);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        return result;

    }

    @Override
    public IssueResult calculateStyle(ReadGraph graph, Resource runtimeDiagram, Resource entry, Resource element, Variable configuration) throws DatabaseException {

        Resource model = Variables.getModel(graph, configuration);
        if (model == null)
            return null;

        List<Resource> contexts = getContexts(graph, element);
        AffineTransform transform = DiagramGraphUtil.getAffineTransform(graph, element);
        Map<Severity, List<Resource>> issuesBySeverity = graph.syncRequest(new ListModelIssuesBySeverity(model, true, true, Severity.NOTE),
                TransientCacheListener.<Map<Severity, List<Resource>>>instance());

        for (Severity severity : Severity.values()) {
            List<Resource> issues = issuesBySeverity.get(severity);
            if (issues != null) {
                Set<Resource> issueContexts = graph.syncRequest(new IssueResourcesContexts(issues));
                if (!Collections.disjoint(issueContexts, contexts))
                    return new IssueResult(severity, transform);
            }
        }

        return null;
    }

    @Override
    public void applyStyleForNode(EvaluationContext observer, INode node, IssueResult result) {
        if (result == null) {
            ProfileVariables.denyChild(node, "", DECORATION_NODE_NAME);
            return;
        }

        SVGNode svgNode = ProfileVariables.claimChild(node, "", DECORATION_NODE_NAME, DecorationSVGNode.class, observer);

        svgNode.setZIndex( Integer.MAX_VALUE );
        svgNode.setTransform(getDecorationPosition(node)); 

        String svgData = svgDataForSeverity(result.getSeverity());
        if (svgData != null)
            svgNode.setData(svgData);
    }

    /**
     * Returns position of the decoration.
     * By default decoration is placed to the top left corner.  Override this method to change the position.
     *  
     * @param node
     * @return
     */
    protected AffineTransform getDecorationPosition(INode node) {
    	Rectangle2D bounds = NodeUtil.getLocalBounds(node, Decoration.class);

        double tx = bounds.getX();
        double ty = bounds.getY();
        return AffineTransform.getTranslateInstance(tx, ty);
    }

    protected String svgDataForSeverity(Severity s) {
        switch (s) {
        case FATAL: return Activator.FATAL_SVG_TEXT;
        case ERROR: return Activator.ERROR_SVG_TEXT;
        case WARNING: return Activator.WARNING_SVG_TEXT;
        case INFO: return Activator.INFO_SVG_TEXT;
        case NOTE: return Activator.NOTE_SVG_TEXT;
        default: return null;
        }
    }

    @Override
    protected void cleanupStyleForNode(INode node) {
        ProfileVariables.denyChild(node, "", DECORATION_NODE_NAME);
    }

    @Override
    public String toString() {
        return "Issue decoration";
    }
    
    /**
     * This is needed to keep the issue decoration up-to-date when its parent
     * element moves.
     */
    public static class IssueResult extends Tuple {
        public IssueResult(Severity severity, AffineTransform transform) {
            super(severity, transform);
        }
        public Severity getSeverity() {
            return (Severity) getField(0);
        }
    }

}


