package org.simantics.modeling.ui.diagram.renaming;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.Configuration;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.services.ComponentNamingStrategy;
import org.simantics.modeling.services.ComponentNamingUtil;
import org.simantics.modeling.services.NamingException;
import org.simantics.modeling.ui.Activator;
import org.simantics.operation.Layer0X;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * @author Hannu Niemist&ouml;
 * @author Tuukka Lehtonen
 */
public class ComponentsRenamingModel {
    public ArrayList<NameEntry> entries = new ArrayList<NameEntry>();
    public Set<NameEntry> selectedEntries = new HashSet<>();
    public String oldNamePrefix;
    public String newNamePrefix;
    public boolean reset;
    public Function1<String, String> prefixValidator;

    private Session session;
    private Variable compositeVariable;
    private Resource configuration;
    private ComponentNamingStrategy namingStrategy;

    public ComponentsRenamingModel read(ReadGraph g, Resource composite) throws DatabaseException {
        this.session = g.getSession();
        this.compositeVariable = Variables.getVariable(g, composite);
        this.configuration = g.syncRequest(new Configuration(composite));

        Layer0 L0 = Layer0.getInstance(g);
        Layer0X L0X = Layer0X.getInstance(g);
        StructuralResource2 STR = StructuralResource2.getInstance(g);
        for(Resource component : g.getObjects(composite, L0.ConsistsOf)) {
            if(!g.isInstanceOf(component, STR.Component))
                continue;
            String name = g.getRelatedValue(component, L0.HasName);
            Resource componentType = g.getPossibleType(component, STR.Component);
            String componentTypePrefix = componentType != null
                    ? g.<String>getPossibleRelatedValue(componentType, L0X.HasGeneratedNamePrefix, Bindings.STRING)
                    : "";
            entries.add(new NameEntry(component, name, name, componentTypePrefix));
        }
        Collections.sort(entries);
        Variable namePrefixValue = compositeVariable.getProperty(g, L0X.HasGeneratedNamePrefix);
        oldNamePrefix = newNamePrefix = namePrefixValue.getValue(g, Bindings.STRING);

        Variable displayValue = namePrefixValue.getPossibleProperty(g, Variables.DISPLAY_VALUE);
        if (displayValue != null)
            prefixValidator = displayValue.getPossiblePropertyValue(g, Variables.INPUT_VALIDATOR);

        this.namingStrategy = ComponentNamingUtil.findNamingStrategy(g, null, composite);

        // By default, select all entries.
        this.selectedEntries.addAll(entries);

        return this;
    }

    public void computeNewNames() {
        final boolean reset = this.reset;
        if (reset) {
            for (NameEntry entry : entries)
                entry.newName = newNamePrefix + entry.namingPrefix;
        } else {
            for (NameEntry entry : entries)
                if (entry.oldName.startsWith(oldNamePrefix))
                    entry.newName = newNamePrefix + entry.oldName.substring(oldNamePrefix.length());
        }

        if (session != null) {
            try {
                session.syncRequest(new ReadRequest() {
                    @Override
                    public void run(ReadGraph graph) throws DatabaseException {
                        validateNewNames(graph, !reset);
                    }
                });
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "validateNewNames failed, see exception for details", e));
            }
        } else {
            if (reset) {
                int i=0;
                for(NameEntry entry : entries)
                    entry.newName = entry.newName + (++i);
            }
        }
    }

    private void validateNewNames(ReadGraph graph, boolean acceptPropositions) throws DatabaseException {
        try {
            if (namingStrategy != null) {
                List<String> propositions = new ArrayList<String>(entries.size());
                for (NameEntry entry : entries)
                     propositions.add(entry.newName);

                propositions = namingStrategy.validateInstanceNames(graph, configuration, propositions, acceptPropositions, null);

                for (int i = 0; i < propositions.size(); ++i) {
                    NameEntry entry = entries.get(i);
                    if (!acceptPropositions || !entry.oldName.equals(entry.newName))
                        entry.newName = propositions.get(i);
                }
            }
        } catch (NamingException e) {
            throw new DatabaseException(e);
        }
    }

    public void write(WriteGraph g) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        Layer0X L0X = Layer0X.getInstance(g);
        for(NameEntry entry : entries)
            if(!entry.oldName.equals(entry.newName) && selectedEntries.contains(entry))
                g.claimLiteral(entry.resource, L0.HasName, entry.newName, Bindings.STRING);
        if(!oldNamePrefix.equals(newNamePrefix))
            compositeVariable.setPropertyValue(g, L0X.HasGeneratedNamePrefix, newNamePrefix, Bindings.STRING);
    }
}
