package org.simantics.modeling.ui.componentTypeEditor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collections;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.swt.widgets.Display;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.scl.SCLDatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.userComponent.ComponentTypeCommands;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.ui.editor.SCLSourceViewerConfigurationNew;
import org.simantics.scl.ui.editor.TextAndErrors;
import org.simantics.structural2.scl.procedural.CompileProceduralComponentTypeRequest;
import org.simantics.structural2.scl.procedural.ProceduralComponentTypeCompilationException;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.logging.TimeLogger;
import org.simantics.utils.ui.SWTUtils;

public class ProceduralComponentTypeCodeDocumentProvider extends SCLModuleEditorDocumentProvider {

    Display display;
    ProceduralComponentTypeCodeEditor editor;

    public ProceduralComponentTypeCodeDocumentProvider(SCLSourceViewerConfigurationNew sourceViever, Display display, ProceduralComponentTypeCodeEditor editor) {
        super(sourceViever);
        this.display = display;
        this.editor = editor;
    }

    @Override
    protected IDocument createDocument(Object element) throws CoreException {
        ResourceEditorInput input = (ResourceEditorInput)element;
        resource = input.getResource();
        try {
            return Simantics.getSession().syncRequest(new UniqueRead<Document>() {
                @Override
                public Document perform(ReadGraph graph) throws DatabaseException {
                    currentText = graph.getValue(resource, Bindings.STRING);
                    errorHappened = false;
                    return new Document(currentText != null ? currentText : "");
                }
            });
        } catch (DatabaseException e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            errorHappened = true;
            return new Document(sw.toString());
        }
    }

    protected void updateAnnotations() {
        Simantics.getSession().asyncRequest(new ResourceRead<TextAndErrors>(resource) {
            @Override
            public TextAndErrors perform(ReadGraph graph) throws DatabaseException {
                Layer0 L0 = Layer0.getInstance(graph);
                String text = graph.getValue(resource, Bindings.STRING);
                Resource componentType = graph.getPossibleObject(resource, L0.PropertyOf);
                CompilationError[] errors = CompilationError.EMPTY_ARRAY;
                if (componentType != null) {
                    try { 
                        graph.syncRequest(new CompileProceduralComponentTypeRequest(componentType));
                    } catch (SCLDatabaseException e) {
                        // Can't compile effectively empty code
                        errors = e.compilationErrors;
                    } catch (ProceduralComponentTypeCompilationException e) {
                        // TODO: this exception is not thrown anywhere. Remove it?
                        errors = e.errors;
                    }
                }
                return new TextAndErrors(text, errors);
            }
        }, new Listener<TextAndErrors>() {
            @Override
            public void execute(final TextAndErrors textAndErrors) {
                if (editor.isDisposed())
                    return;
                SWTUtils.asyncExec(display, new Runnable() {
                    @Override
                    public void run() {
                        if (editor.isDisposed())
                            return;
                        if(textAndErrors.errors.length > 0) {
                            setAnnotations(Arrays.asList(textAndErrors.errors));
                        } else {
                            setAnnotations(Collections.<CompilationError>emptyList());
                        }
                    }
                });
            }

            @Override
            public void exception(Throwable t) {
                Activator.getDefault().getLog().log(
                        new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Internal error in procedural user component code compilation.", t));
            }

            @Override
            public boolean isDisposed() {
                return editor.isDisposed();
            }
        });
    }

    @Override
    protected void doSaveDocument(IProgressMonitor monitor, Object element,
            IDocument document, boolean overwrite) throws CoreException {
        TimeLogger.resetTimeAndLog(getClass(), "doSaveDocument");
        currentText = document.get();
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                ComponentTypeCommands.saveProceduralCode(graph, resource, currentText);
            }
        });
    }

}
