/*******************************************************************************
 * Copyright (c) 2013, 2015 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - adaption to SCLModuleEditor/TextEditor
 *******************************************************************************/
package org.simantics.modeling.ui.componentTypeEditor;

import java.util.function.Supplier;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.combinations.Combinators;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.scl.ui.editor.SCLSourceViewerConfigurationNew;
import org.simantics.scl.ui.editor.SharedTextColorsNew;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.TitleUpdater;
import org.simantics.ui.workbench.ToolTipRequest;

/**
 * @author Hannu Niemist&ouml;
 * @author Tuukka Lehtonen (extended from SCLModuleEditor)
 */
public class ComponentTypeScriptEditor extends SCLModuleEditor {

    public ComponentTypeScriptEditor() {
        super();
    }

    @Override
    protected void preInitialize() {
        setDocumentProvider(new ComponentTypeScriptDocumentProvider(this));
        Display display = PlatformUI.getWorkbench().getDisplay();
        SCLSourceViewerConfigurationNew sourceViewerConfiguration = new SCLSourceViewerConfigurationNew(resourceManager);
        setSourceViewerConfiguration(sourceViewerConfiguration);
        
    }
    
    protected ParametrizedRead<IResourceEditorInput, Boolean> getInputValidator() {
        return new ParametrizedRead<IResourceEditorInput, Boolean>() {
            @Override
            public Read<Boolean> get(IResourceEditorInput parameter) {
                return Combinators.constant(Boolean.TRUE);
            }
        };
    }
    
    @Override
    protected void updatePartName() {
        setPartName(getEditorInput().getName());

        Session session = Simantics.peekSession();
        if (session != null) {
            Supplier<Boolean> disposedCallback = this::isDisposed;
            session.asyncRequest(
                    new UniqueRead<String>() {
                        @Override
                        public String perform(ReadGraph graph)
                                throws DatabaseException {
                            Layer0 L0 = Layer0.getInstance(graph);
                            StructuralResource2 STR = StructuralResource2.getInstance(graph);
                            Resource script = getResourceInput().getResource();
                            String name = graph.getRelatedValue(script, L0.HasName);
                            Resource componentType = graph.getSingleObject(script, STR.ComponentType_hasScript_Inverse);
                            String ctName = graph.getRelatedValue(componentType, L0.HasName);
                            return ctName + " " + name;
                        }
                        
                    },
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setPartName, disposedCallback));
            session.asyncRequest(
                    new ToolTipRequest(getSite().getId(), getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setTitleToolTip, disposedCallback));
        }
    }
    
    private static final String[] EXECUTION_PHASES = new String[] {
        "step",
        "analogAutomation",
        "binaryAutomation",
        "preparation"
    };
    
    private static final String[] EXECUTION_PHASE_LABELS = new String[] {
        "Execute at each step",
        "Execute together with analog automation",
        "Execute together with binary automation",
        "Execute during preparation"
    };
    
    @Override
    public void createPartControl(Composite parent) {
        GridLayoutFactory.fillDefaults().applyTo(parent);
        
        final CCombo combo = new CCombo(parent, SWT.READ_ONLY | SWT.BORDER);
        for(String label : EXECUTION_PHASE_LABELS)
            combo.add(label);
        Session session = Simantics.peekSession();
        final Resource script = getResourceInput().getResource();
        if(session != null) {
            session.asyncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    StructuralResource2 STR = StructuralResource2.getInstance(graph);
                    final String type = graph.getPossibleRelatedValue(script, STR.ComponentTypeScript_type);
                    if(type != null)
                        combo.getDisplay().asyncExec(new Runnable() {
                            @Override
                            public void run() {
                                for(int i=0;i<EXECUTION_PHASES.length;++i)
                                    if(EXECUTION_PHASES[i].equals(type)) {
                                        combo.select(i);
                                        return;
                                    }
                            }
                        });
                }
            });
        }
        combo.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                final int id = combo.getSelectionIndex();
                Simantics.getSession().asyncRequest(new WriteRequest() {
                    @Override
                    public void perform(WriteGraph graph)
                            throws DatabaseException {
                        StructuralResource2 STR = StructuralResource2.getInstance(graph);
                        String currentType = graph.getPossibleRelatedValue(script, STR.ComponentTypeScript_type);
                        String newType = EXECUTION_PHASES[id];
                        if(!newType.equals(currentType))
                            graph.claimLiteral(script, STR.ComponentTypeScript_type, newType, Bindings.STRING);
                    }
                });
            }
        });
        GridDataFactory.fillDefaults().grab(true, false).applyTo(combo);
        
        Composite editorComposite = new Composite(parent, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(editorComposite);
        editorComposite.setLayout(new FillLayout());
        super.createPartControl(editorComposite);
    }

}