/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.gfx.clipboard;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.ByteArrayTransfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.gfx.clipboard.headers.Win32Constants;
import org.simantics.utils.ui.gfx.clipboard.streams.AddBitmapHeaderInputStream;
import org.simantics.utils.ui.gfx.clipboard.streams.RemoveBitmapHeaderOutputStream;
import org.simantics.utils.ui.gfx.clipboard.streams.UncompressedDIBInputStream;


/**
 * Transfer class for handing image cut/copy/paste operations between Eclipse
 * and native WIN32 apps
 * 
 * @author Marko Luukkainen
 *
 */
public class ImageDataTransfer extends ByteArrayTransfer {

    private static final ImageDataTransfer instance = new ImageDataTransfer();

    private static final String CF_DIB = "CF_DIB";

    private ImageDataTransfer() {
        //
    }

    public static ImageDataTransfer getInstance() {
        return ImageDataTransfer.instance;
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.dnd.Transfer#getTypeIds()
     */
    @Override
    protected final int[] getTypeIds() {
        return new int[] { Win32Constants.CF_DIB };
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.dnd.Transfer#getTypeNames()
     */
    @Override
    protected final String[] getTypeNames() {
        return new String[] { ImageDataTransfer.CF_DIB };
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.dnd.ByteArrayTransfer#nativeToJava(org.eclipse.swt.dnd.TransferData)
     */
    @Override
    protected Object nativeToJava(TransferData transferData) {
        final Object o = super.nativeToJava(transferData);
        final byte[] bytes = (byte[]) o;

        try {
            final InputStream iStream = new AddBitmapHeaderInputStream(
                    new UncompressedDIBInputStream(new ByteArrayInputStream(
                            bytes)));
            final ImageData[] data = new ImageLoader().load(iStream);
            if (data.length <= 0) {
                return null;
            }
            return data[0];
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Cannot get image from clipboard", e);
            return null;
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.dnd.ByteArrayTransfer#javaToNative(java.lang.Object, org.eclipse.swt.dnd.TransferData)
     */
    @Override
    protected void javaToNative(Object object, TransferData transferData) {
        final ImageData imgData = (ImageData) object;
        final ImageLoader loader = new ImageLoader();
        final ByteArrayOutputStream oStream = new ByteArrayOutputStream();
        final byte[] bytes;

        loader.data = new ImageData[] { imgData };
        loader.save(new RemoveBitmapHeaderOutputStream(oStream), SWT.IMAGE_BMP);
        bytes = oStream.toByteArray();
        super.javaToNative(bytes, transferData);
    }
}
