/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/

package org.simantics.utils.threads.ua;

import java.util.EnumSet;

/**
 * State of an async execution (=work)
 * 
 * Initial states: Ready
 * Final states: Complete, Canceled, Error, Interrupted
 * 
 * State transitions:
 *   Ready -> Canceled
 *   Ready -> Working -> Complete
 *   Ready -> Working -> Error 
 *   Ready -> Working -> Interrupting -> Error
 *   Ready -> Working -> Interrupting -> Interrupted
 * 
 * @see WorkMonitor
 * @see Work
 * @see AsyncRun
 * @author Toni Kalajainen (toni.kalajainen@vtt.fi)
 */
public enum WorkState {

	Ready,			// Work has not been started. (Initial State)
	Working,		// The work has started.
	Complete,		// The work has completed. (Final State)
	Interrupting,	// Interrupt signal has been sent.
	Interrupted,	// The work was has ended after interrupt signal. (Final State)
	Error,			// There was an error during the execution. (Final State)
	Canceled;		// Work was canceled before it was started. (Final State)
	
	public static final EnumSet<WorkState> READY_STATE = EnumSet.of(WorkState.Ready);
	public static final EnumSet<WorkState> WORKING_STATE = EnumSet.of(WorkState.Working);
	public static final EnumSet<WorkState> FINAL_STATES = EnumSet.of(WorkState.Complete, WorkState.Error, WorkState.Canceled, WorkState.Interrupted);
	public static final EnumSet<WorkState> COMPLETE_STATE = EnumSet.of(WorkState.Complete);
	public static final EnumSet<WorkState> ERROR_STATE = EnumSet.of(WorkState.Error);
	public static final EnumSet<WorkState> CANCELED_STATE = EnumSet.of(WorkState.Canceled);
	public static final EnumSet<WorkState> CANCELED_STATES = EnumSet.of(WorkState.Canceled, WorkState.Interrupted, WorkState.Interrupting);

	public boolean isFinalState()
	{
		return FINAL_STATES.contains(this);
	}
	
	public boolean isInitialState()
	{
		return this == Ready;
	}
	
}
