/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.bytes;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.simantics.utils.strings.FileNameUtils;


/**
 * Byte array operations
 * 
 * @author Toni Kalajainen
 */
public class ByteArrays {

    /**
     * Copy the contents of source array to dst array
     * @param src
     * @param dest
     */
    public static void move(byte[] src, byte[] dest)
    {
        System.arraycopy(src, 0, dest, 0, src.length);
    }

    public static byte[] mid(byte[] src, int pos, int len)
    {
        if (pos+len > src.length)
            throw new IndexOutOfBoundsException();

        byte result[] = new byte[len];
        System.arraycopy(src, pos, result, 0, len);
        return result;
    }

    /**
     * copy byte array into another starting from the given position.
     * @param src
     * @param dest
     * @param pos
     * @param length
     */
    public static void move(byte[] src, byte[] dest, int pos, int length)
    {
        System.arraycopy(src, 0, dest, pos, length);
    }

    /**
     * Merge two arrays into one
     * @param src
     * @param src2
     * @return
     */
    public static byte[] merge(byte[] src, byte[] src2)
    {
        byte result[] = new byte[src.length+src2.length];
        System.arraycopy(src, 0, result, 0, src.length);
        System.arraycopy(src2, 0, result, src.length, src2.length);
        return result;
    }


    /**
     * Merge 3 arrays into one
     * @param src
     * @param src2
     * @param src3
     * @return
     */
    public static byte[] merge(byte[] src, byte[] src2, byte[] src3)
    {
        byte result[] = new byte[src.length+src2.length+src3.length];
        System.arraycopy(src, 0, result, 0, src.length);
        System.arraycopy(src2, 0, result, src.length, src2.length);
        System.arraycopy(src3, 0, result, src.length+src2.length, src3.length);
        return result;
    }


    /**
     * Merge 4 arrays into one
     * @param src
     * @param src2
     * @param src3
     * @param src4
     * @return
     */
    public static byte[] merge(byte[] src, byte[] src2, byte[] src3, byte[] src4)
    {
        byte result[] = new byte[src.length+src2.length+src3.length+src4.length];
        System.arraycopy(src, 0, result, 0, src.length);
        System.arraycopy(src2, 0, result, src.length, src2.length);
        System.arraycopy(src3, 0, result, src.length+src2.length, src3.length);
        System.arraycopy(src4, 0, result, src.length+src2.length+src3.length, src4.length);
        return result;
    }

    /**
     * Merge 5 arrays into one
     * @param src
     * @param src2
     * @param src3
     * @param src4
     * @param src5
     * @return
     */
    public static byte[] merge(byte[] src, byte[] src2, byte[] src3, byte[] src4, byte[] src5)
    {
        byte result[] = new byte[src.length+src2.length+src3.length+src4.length+src5.length];
        System.arraycopy(src, 0, result, 0, src.length);
        System.arraycopy(src2, 0, result, src.length, src2.length);
        System.arraycopy(src3, 0, result, src.length+src2.length, src3.length);
        System.arraycopy(src4, 0, result, src.length+src2.length+src3.length, src4.length);
        System.arraycopy(src5, 0, result, src.length+src2.length+src3.length+src4.length, src5.length);
        return result;
    }

    /**
     * Swap Little-endian and Big-endian integer byte order
     * @param i
     * @return
     */
    public static int swap(int i) {
        int byte0 = i & 0xff;
        int byte1 = (i>>8) & 0xff;
        int byte2 = (i>>16) & 0xff;
        int byte3 = (i>>24) & 0xff;
        // swap the byte order
        return (byte0<<24) | (byte1<<16) | (byte2<<8) | byte3;
    }

    public static void saveToFile(byte data[], String filename)
    throws IOException
    {
        // Create directories
        String dir = FileNameUtils.extractFileDir(filename);
        (new File(dir)).mkdirs();
        FileOutputStream out = new FileOutputStream(filename);
        out.write(data);
        out.flush();
        out.close();
    }

    /**
     * Test cases
     * 
     * @param args
     */
    public static void main(String[] args) {
        byte array1[] = BEInt.toBytes(5);
        byte array2[] = BEInt.toBytes(10);
        byte array[] = merge(array1, array2);
        printByteArray(array);
        System.out.println();
    }

    public static void printByteArray(byte array[]) {
        for (int i=0; i<array.length; i++) {
            System.out.print(array[i] & 0xff);
            if (i<array.length-1)
                System.out.print(",");
        }
    }
}
