/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.utils.datastructures.prioritystack;

import org.simantics.utils.threads.IThreadWorkQueue;


/**
 * Stack of interactor layers
 * @param <E> type of stack items
 */
public interface IPriorityStack<E> {
	
	/**
	 * Put an item to the stack 
	 * to the top of the stack.
	 * @param item
	 * @param priority 
	 */
	void add(E item, int priority);

	/**
	 * Get priority of an item.
	 * Priority affects the event handing, hint overriding and to the paint order of 
	 * the interactor.
	 * 
	 * @param item the item
	 * @return priority of the interactor or <code>null</code> if item does not exist 
	 */
	Integer getPriority(E item);
	
	/**
	 * Remove item
	 * @param item
	 * @return true if was found and removed
	 */
	boolean remove(E item); 

	/**
	 * Checks whether an item is in the stack
	 * @param item
	 * @return true if stack contains the specified item
	 */
	boolean contains(E item);

    /**
     * Get all items by class. All items assignable to the specified class are
     * returned.
     * 
     * @param <R>
     * @param clazz
     * @return all items assignable to the specified class
     */
	<R extends E> R[] getItemsByClass(Class<R> clazz);

    /**
     * Get a single item by class. (Convenience method)
     * 
     * @param <R>
     * @param clazz
     * @return a single item of the specified class
     * @throws RuntimeException if single item was not found or if multiple
     *         matching items were found
     */
	<R extends E> R getSingleItem(Class<R> clazz);
	
	/**
	 * Get a snapshot of all the items
	 * @return ordered list of items. The highest priority item is the last element.
	 */
	E[] toArray();
	
	/**
	 * add listener
	 * @param thread
	 * @param listener
	 */
	void addStackListener(IThreadWorkQueue thread, IPriorityStackListener<E> listener);
	
	/**
	 * remove listener
	 * @param thread
	 * @param listener
	 */
	void removeStackListener(IThreadWorkQueue thread, IPriorityStackListener<E> listener);

	/**
	 * add listener
	 * @param listener
	 */
	void addStackListener(IPriorityStackListener<E> listener);
	
	/**
	 * remove listener
	 * @param listener
	 */
	void removeStackListener(IPriorityStackListener<E> listener);
	
}
