/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/

package org.simantics.utils.datastructures.context;

import java.util.Collection;

import org.simantics.utils.threads.IThreadWorkQueue;

/**
 * Observable container.
 * 
 * TODO: add addAll() method
 * 
 * @see Context for implementation
 * 
 * @author Toni Kalajainen
 * @param <E>
 */
public interface IContext<E> {
	
	/**
	 * Put an item to the context. 
	 * @param item
	 */
	void add(E item);
	
	/**
	 * Remove an item from the context.
	 * @param item 
	 * @return true if was found and removed
	 */
	boolean remove(E item); 

	/**
	 * Checks whether an item is in the context
	 * @param item an item
	 * @return true if context contains an item
	 */
	boolean contains(E item);

	/**
	 * Clears all items from the context.
	 */
	void clear();

	/**
	 * Get all items by class
     * @param <R>
	 * @param clazz
	 * @return collection 
	 */
	<R extends E> Collection<R> getItemsByClass(Class<R> clazz);
	
	/**
	 * Get a single item by class. 
	 * (Convenience method)
	 * 
	 * @param <R>
	 * @param clazz
	 * @return the item 
	 * @throws RuntimeException if single interactor was not found
	 */
	<R extends E> R getSingleItem(Class<R> clazz);
	
	/**
	 * Checks that the context has the item by the class
	 * @param <R>
	 * @param clazz
	 * @return <code>true</code> if contained
	 */
	<R> boolean containsItemByClass(Class<R> clazz);

	/**
	 * Get a single item by class. 
	 * (Convenience method)
	 * 
	 * @param <R>
	 * @param clazz
	 * @return the item or <code>null</code>
	 * @throws RuntimeException if more than one items were found
	 */
	<R extends E> R getAtMostOneItemOfClass(Class<R> clazz);
	
	/**
	 * Get a snapshot of all the items
	 * @return ordered list of items. The highest priority item is the last element.
	 */
	E[] toArray();
	
	/**
	 * add listener
	 * @param listener
	 */
	void addContextListener(IContextListener<E> listener);
	
	/**
	 * remove listener
	 * @param listener
	 */
	void removeContextListener(IContextListener<E> listener);
	
	/**
	 * add listener
     * @param thread thread
	 * @param listener
	 */
	void addContextListener(IThreadWorkQueue thread, IContextListener<E> listener);
	
	/**
	 * remove listener
	 * @param thread thread
	 * @param listener
	 */
	void removeContextListener(IThreadWorkQueue thread, IContextListener<E> listener);
	
}
