/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.editor;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.WorkbenchException;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.ui.workbench.ResourceViewPartUtils;
import org.simantics.ui.workbench.action.ResourceEditorAdapterAction;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.action.IPriorityAction;
import org.simantics.utils.ui.action.PriorityActionAdapter;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * A base implementation for {@link EditorAdapter} that adapts
 * {@link #canHandle(ReadGraph, Object)} into
 * {@link #canHandle(ReadGraph, Resource)} and {@link #openEditor(Object)} into
 * {@link #openEditor(Resource)}.
 * 
 * If you only plan to support {@link Resource} or {@link Resource}-adaptable
 * editor inputs, this class is a bit easier to use than
 * {@link AbstractEditorAdapter}.
 * 
 * @author Tuukka Lehtonen
 */
public abstract class AbstractResourceEditorAdapter extends AbstractEditorAdapter {

    public AbstractResourceEditorAdapter(String name) {
        super(name, null, 0);
    }

    public AbstractResourceEditorAdapter(String name, ImageDescriptor icon) {
        super(name, icon, 0);
    }

    public AbstractResourceEditorAdapter(String name, ImageDescriptor icon, int priority) {
        super(name, icon, priority);
    }

    /**
     * Override to perform the can handle check based on a single resource
     * adapted from the Object input using default mechanisms provided by
     * {@link ResourceAdaptionUtils}.
     * 
     * Returns <code>false</code> by default.
     * 
     * @param g
     * @param input
     * @return
     * @throws DatabaseException
     */
    protected boolean canHandle(ReadGraph g, Resource input) throws DatabaseException {
        return false;
    }

    /**
     * Override to open an editor based on a single resource adapted from the
     * Object input using default mechanisms provided by
     * {@link ResourceAdaptionUtils}.
     * 
     * @param input
     * @throws Exception
     */
    protected void openEditor(Resource input) throws Exception {
    }

    @Override
    public boolean canHandle(ReadGraph g, Object input) throws DatabaseException {
        Resource r = ResourceAdaptionUtils.toSingleResource(input);
        if (r == null)
            return false;
        return canHandle(g, r);
    }

    @Override
    public void openEditor(Object input) throws Exception {
        Resource r = ResourceAdaptionUtils.toSingleResource(input);
        if (r != null)
            openEditor(r);
    }

    protected void openViewWithId(String viewId, Resource id, String suffix) throws Exception {
        Session ls = SimanticsUI.getSession();
        ResourceViewPartUtils.activateViewForResource(viewId, ls, id, null);
    }

    protected void openViewWithId(String viewId, Resource id) throws Exception {
        openViewWithId(viewId, id, null);
    }

    protected void openViewWithIdInPerspective(String viewId, Resource id, String perspectiveId) throws Exception {
        try {
            WorkbenchUtils.showPerspective(perspectiveId);
        } catch (WorkbenchException e) {
            ErrorLogger.getDefault().logError("Could not open perspective with ID'" + perspectiveId + "'.", e);
        }

        openViewWithId(viewId, id);
    }

    public IAction toAction(Resource r) {
        return new ResourceEditorAdapterAction(this, r);
    }

    public IPriorityAction toPriorityAction(int priority, Resource r) {
        return new PriorityActionAdapter(priority, toAction(r));
    }

}