package org.simantics.ui.workbench.e4;

import java.util.List;
import java.util.Map;

import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.model.application.ui.MUIElement;
import org.eclipse.e4.ui.model.application.ui.advanced.MArea;
import org.eclipse.e4.ui.model.application.ui.advanced.MPlaceholder;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.model.application.ui.basic.MPartSashContainerElement;
import org.eclipse.e4.ui.model.application.ui.basic.MPartStack;
import org.eclipse.e4.ui.workbench.modeling.EModelService;
import org.eclipse.e4.ui.workbench.modeling.EPartService;
import org.eclipse.e4.ui.workbench.modeling.EPartService.PartState;
import org.eclipse.ui.PlatformUI;
import org.simantics.db.Resource;
import org.simantics.utils.datastructures.ArrayMap;

/**
 * @author Tuukka Lehtonen
 *
 */
public class E4WorkbenchUtils {

    private static final String[] KEYS_RESOURCE = { E4ResourceEditorConstants.KEY_RESOURCE };

    /**
     * 
     */
    private static final String TAG_EDITOR = "Editor";

    /**
     * The Legacy IDE application ID for the "editor area" partstack. In E3
     * compatibility mode legacy application the {@link MPartStack} lies under
     * the shared {@link MArea} (org.eclipse.ui.editorss).
     * 
     * <p>
     * In a pure E4 application this needs to be changed.
     */
    public static final String EDITOR_PART_STACK_ID = "org.eclipse.e4.primaryDataStack";

    public static final String EDITOR_AREA_ID = "org.eclipse.ui.editorss";

    /**
     * @param editorId
     * @param contributionURI
     * @param iconURI
     * @param inputId
     * @param inputMap
     */
    public static MPart openEditor(String editorId, String contributionURI, String iconURI, String inputId, Map<String, Object> inputMap) {
        if (editorId == null)
            throw new NullPointerException("null editor ID");
        if (contributionURI == null)
            throw new NullPointerException("null contributionURI");
        if (inputId == null)
            throw new NullPointerException("null input ID");
        if (inputMap == null)
            throw new NullPointerException("null input map");

        IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        EModelService modelService = context.get(EModelService.class);
        EPartService partService = context.get(EPartService.class);
        MApplication app = context.get(MApplication.class);

        String editorID = editorId + "_" + inputId;

        List<MPart> parts = modelService.findElements(app, editorID, MPart.class, null);
        MPart editor = null;
        if (parts == null || parts.isEmpty()) {
            editor = modelService.createModelElement(MPart.class);
            editor.setElementId(editorID);
            editor.setContributionURI(contributionURI);
            if (iconURI != null)
                editor.setIconURI(iconURI);
            editor.setCloseable(true);
            editor.getTags().add(TAG_EDITOR);
            editor.getTransientData().putAll(inputMap);

            MPartStack stack = getEditorPartStack(modelService, app);
            stack.getChildren().add(editor);
        } else {
            editor = parts.get(0);
        }

        partService.showPart(editor, PartState.ACTIVATE);
        return editor;
    }

    /**
     * @param editorId
     * @param contributionURI
     * @param iconURI
     * @param input
     */
    public static MPart openEditor(String editorId, String contributionURI, String iconURI, Resource input) {
        String inputId = Long.toString(input.getResourceId());
        Map<String, Object> inputMap = ArrayMap.make(KEYS_RESOURCE, input);
        return openEditor(editorId, contributionURI, iconURI, inputId, inputMap);
    }

    /**
     * @param editorElementId the unique element ID of the part (editor)
     */
    public static void closeEditor(String editorElementId) {
        IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        EModelService modelService = context.get(EModelService.class);
        MApplication app = context.get(MApplication.class);
        modelService
        .findElements(app, editorElementId, MPart.class, null)
        .forEach(p -> p.getContext().get(EPartService.class).hidePart(p));
    }

    public static void activatePart(MPart part) {
        EPartService partService = part.getContext().get(EPartService.class);
        partService.activate(part);
    }

    public static MPartStack getEditorPartStack(EModelService modelService, MApplication app) {
        MPartStack stack = (MPartStack) modelService.find(E4WorkbenchUtils.EDITOR_PART_STACK_ID, app);
        if (stack == null) {
            MArea editorArea = null;
            MUIElement area = modelService.find(E4WorkbenchUtils.EDITOR_AREA_ID, app);
            if (area instanceof MPlaceholder) {
                editorArea = (MArea) ((MPlaceholder) area).getRef();
            } else if (area instanceof MArea) {
                editorArea = (MArea) area;
            }
            if (editorArea == null)
                throw new IllegalStateException("Shared editor area (" + E4WorkbenchUtils.EDITOR_AREA_ID + ") not found in application model");
            for (MPartSashContainerElement container : editorArea.getChildren()) {
                if (container instanceof MPartStack) {
                    stack = (MPartStack) container;
                    break;
                }
            }
        }
        if (stack == null)
            throw new IllegalStateException("Could not find part stack under editor area.");
        return stack;
    }

    public static void openAndShowPart(MPart part) {
        IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        EPartService partService = context.get(EPartService.class);
        if (!partService.isPartVisible(part))
            partService.showPart(part, PartState.ACTIVATE);
        else
            partService.activate(part);
    }

    
    public static void openAndShowPart(String partId) {
        IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        EPartService partService = context.get(EPartService.class);
        partService.showPart(partId, PartState.ACTIVATE);
    }

    public static MPart getMPartById(String partId) {
        IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
        EPartService partService = context.get(EPartService.class);
        return partService.findPart(partId);
    }

}
