/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.dialogs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.dialogs.FilteredItemsSelectionDialog;
import org.eclipse.ui.dialogs.SearchPattern;
import org.simantics.db.Resource;
import org.simantics.db.exception.InvalidResourceReferenceException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.internal.Activator;
import org.simantics.utils.datastructures.Pair;

public abstract class ResourceSelectionDialog3<T> extends FilteredItemsSelectionDialog {

    private final Map<T, Pair<String, ImageDescriptor>> contentMap;
    private final String title;
    private LocalResourceManager resourceManager;

    class ResourceSelectionHistory extends FilteredItemsSelectionDialog.SelectionHistory {

        @Override
        protected Object restoreItemFromMemento(IMemento memento) {
            // FIXME: somehow create a collective transaction inside which the Graph.getRandomAccessReference should be invoked.
//            Resource r = SimanticsUI.getSession().getRandomAccessReference(memento.getTextData());
//            return r;
            return null;
        }

        @Override
        protected void storeItemToMemento(Object item, IMemento memento) {
            if(item instanceof Resource) {
                try {
                    SerialisationSupport support = SimanticsUI.getSession().getService(SerialisationSupport.class);
                    memento.putTextData(support.getResourceSerializer().createRandomAccessId((Resource)item));
                } catch (InvalidResourceReferenceException e) {
                    e.printStackTrace();
                }
            }
        }
        
    };

    public ResourceSelectionDialog3(Shell shell, Map<T, Pair<String, ImageDescriptor>> parameter, String title) {
        this(shell, parameter, title, true);
    }

    @Override
    protected Control createContents(Composite parent) {
        this.resourceManager = new LocalResourceManager(JFaceResources.getResources(), parent);
        return super.createContents(parent);
    }
    
    public ResourceSelectionDialog3(Shell shell, Map<T, Pair<String, ImageDescriptor>> parameter, String title, boolean multi) {
        
        super(shell, multi);
        this.contentMap = parameter;
        this.title = title;

        ILabelProvider labelProvider = new LabelProvider() {
            @Override
            public String getText(Object element) {
                Pair<String, ImageDescriptor> pair = contentMap.get(element);
                if(pair != null) return pair.first;
                else return null;
            }
            @Override
            public Image getImage(Object element) {
                Pair<String, ImageDescriptor> pair = contentMap.get(element);
                if(pair != null && pair.second != null) return resourceManager.createImage(pair.second);
                else return null;
            }
        }; 

        setListLabelProvider(labelProvider);
        setDetailsLabelProvider(labelProvider);

        setSelectionHistory(new ResourceSelectionHistory());
        setTitle(title);
        
    }

    @Override
    protected Control createExtendedContentArea(Composite parent) {
        // Don't create anything extra at the moment
        return null;
    }

    class ResourceSelectionDialogItemsFilter extends FilteredItemsSelectionDialog.ItemsFilter {
        
        public ResourceSelectionDialogItemsFilter() {
            String patternText = getPattern();
            patternMatcher = new SearchPattern();
            if(patternText != null && patternText.length() > 0)
                patternMatcher.setPattern(patternText);
            else 
                patternMatcher.setPattern("*");
        }

        @Override
        public boolean isConsistentItem(Object item) {
            return true;
        }

        @Override
        public boolean matchItem(Object item) {
            return matches(contentMap.get(item).first);
        }
    }

    @Override
    protected ItemsFilter createFilter() {
        return new ResourceSelectionDialogItemsFilter();
    }

    @Override
    protected void fillContentProvider(AbstractContentProvider contentProvider,
            ItemsFilter itemsFilter, IProgressMonitor progressMonitor) throws CoreException {
        for(T o : contentMap.keySet())
            contentProvider.add(o, itemsFilter);
        if (progressMonitor != null)
            progressMonitor.done();
    }

    protected abstract IDialogSettings getBaseDialogSettings();

    @Override
    protected IDialogSettings getDialogSettings() {
        IDialogSettings base = getBaseDialogSettings();
        if (base == null) base = Activator.getDefault().getDialogSettings();
        IDialogSettings settings = base.getSection(title);
        if (settings == null)
            settings = base.addNewSection(title);
        return settings;
    }

    @Override
    public String getElementName(Object item) {
        return contentMap.get(item).first;
    }

    @Override
    protected Comparator<?> getItemsComparator() {
        return new Comparator<Object>() {

            @Override
            public int compare(Object o1, Object o2) {
                return contentMap.get(o1).first.compareToIgnoreCase(contentMap.get(o2).first);
            }
            
        };
    }

    @Override
    protected IStatus validateItem(Object item) {
        return Status.OK_STATUS;
    }

    /**
     * Made publicly visible.
     * @see org.eclipse.ui.dialogs.FilteredItemsSelectionDialog#updateStatus(org.eclipse.core.runtime.IStatus)
     */
    @Override
    public void updateStatus(IStatus status) {
        super.updateStatus(status);
    }

    @SuppressWarnings("unchecked")
	public Collection<T> getResultT() {
    	Object[] res = getResult();
    	if(res == null) return Collections.emptyList();
    	ArrayList<T> result = new ArrayList<T>();
    	for(Object o : res) {
    		result.add((T)o);
    	}
    	return result;
    }
    
}
