/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.spreadsheet.ui;

import java.awt.Component;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.util.EventObject;

import javax.swing.DefaultCellEditor;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.spreadsheet.CellEditor;
import org.simantics.spreadsheet.ClientModel;
import org.simantics.spreadsheet.util.SpreadsheetUtils;

class TextTableCellEditor extends DefaultCellEditor implements SpreadsheetCellEditor {

    final int row;
    final int column;
    final CellEditor<?> editor;
    final String initial;
    
    private static final long serialVersionUID = 8039248821751464832L;

    JTextField ftf;
    ClientModel clientModel;
	
    public TextTableCellEditor(final int row, final int column,final String initial, final CellEditor<?> editor, ClientModel clientModel) {
    	
        super(new JTextField());
        
    	assert(editor != null);

        this.row = row;
        this.column = column;
        this.initial = initial == null ? "" : initial;
        this.editor = editor;
        this.clientModel = clientModel;
        
        ftf = (JTextField)getComponent();

        ftf.setHorizontalAlignment(JTextField.TRAILING);
        
    }

    @Override
    public boolean isCellEditable(EventObject event) {
        if (super.isCellEditable(event)) {
            if (event instanceof KeyEvent) {
                KeyEvent ke = (KeyEvent) event;
                int kc = ke.getKeyCode();
                switch (kc) {
                case KeyEvent.VK_ESCAPE:
                    return false;
                case KeyEvent.VK_F2:
                    return true;
                }
                if (ke.isActionKey())
                    return false;
            }
            return true;
        }
        return false;
    }

    //Override to invoke setValue on the formatted text field.
    public Component getTableCellEditorComponent(JTable table,
            Object value, boolean isSelected,
            int row, int column) {
    	JTextField ftf =
            (JTextField)super.getTableCellEditorComponent(
                table, value, isSelected, row, column);
        ftf.setText(initial);
        ftf.selectAll();
        ftf.setFont(ftf.getFont().deriveFont(AffineTransform.getScaleInstance(0.8, 0.8)));
        return ftf;
    }

    //Override to ensure that the value remains an Integer.
    public Object getCellEditorValue() {
    	JTextField ftf = (JTextField)getComponent();
        return ftf.getText();
    }

    /** 
     * Lets the user know that the text they entered is 
     * bad. Returns true if the user elects to revert to
     * the last good value.  Otherwise, returns false, 
     * indicating that the user wants to continue editing.
     */
    protected boolean userSaysRevert() {
        Toolkit.getDefaultToolkit().beep();
        ftf.selectAll();
        Object[] options = {"Edit",
                            "Revert"};
        int answer = JOptionPane.showOptionDialog(
            SwingUtilities.getWindowAncestor(ftf),
            "The value must be an integer between "
//            + minimum + " and "
//            + maximum + ".\n"
            + "You can either continue editing "
            + "or revert to the last valid value.",
            "Invalid Text Entered",
            JOptionPane.YES_NO_OPTION,
            JOptionPane.ERROR_MESSAGE,
            null,
            options,
            options[1]);
	    
        if (answer == 1) { //Revert!
            ftf.setText(ftf.getText());
	    return true;
        }
	return false;
    }

    @Override
    public void commit() {
        
    	String str = (String)getCellEditorValue();

    	String cellName = SpreadsheetUtils.cellName(row, column);
    	String expression = clientModel.getPossiblePropertyAt(cellName, ClientModel.CONTENT_EXPRESSION);
		if(expression == null) {
			Variant content = SpreadsheetUtils.getSafeClientVariant(clientModel, cellName, ClientModel.CONTENT);
			if(content != null)
				expression = content.getValue().toString();
			else
				expression = "";
		} else {
			expression = "=" + expression;
		}        
        if (expression.equals(str))
        	return;
        
        if (str.startsWith("=")) {
			editor.edit(null, SpreadsheetUtils.cellName(row, column), ClientModel.CONTENT_EXPRESSION, str, Bindings.STRING, null);
		} else {
			editor.edit(null, SpreadsheetUtils.cellName(row, column), Variant.ofInstance(str), null);
		}
        
    }
    
}
