/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics;

import java.util.Properties;

import org.apache.log4j.Hierarchy;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.spi.LoggerFactory;
import org.apache.log4j.spi.RootLogger;

/**
 * This class encapsulates a Log4J Hierarchy and centralizes all Logger access.
 */
public class LogManager {

    private Hierarchy hierarchy;

    /**
     * Creates a new LogManager. Saves the log and state location.
     * Creates a new Hierarchy and add a new EventListener to it.
     * Configure the hierarchy with the properties passed. Add this object to
     * the list of active log managers.
     * 
     * @param properties log configuration properties
     */
    public LogManager(Properties properties) {
        this.hierarchy = new Hierarchy(new RootLogger(Level.DEBUG));
        new PropertyConfigurator().doConfigure(properties, this.hierarchy);
    }

    /**
     * Checks if this PluginLogManager is disabled for this level.
     * 
     * @param level level value
     * @return boolean true if it is disabled
     */
    public boolean isDisabled(int level) {
        return this.hierarchy.isDisabled(level);
    }

    /**
     * Enable logging for logging requests with level l or higher. By default
     * all levels are enabled.
     * 
     * @param level level object
     */
    public void setThreshold(Level level) {
        this.hierarchy.setThreshold(level);
    }

    /**
     * The string version of setThreshold(Level level)
     * 
     * @param level level string
     */
    public void setThreshold(String level) {
        this.hierarchy.setThreshold(level);
    }

    /**
     * Get the repository-wide threshold.
     * 
     * @return Level
     */
    public Level getThreshold() {
        return this.hierarchy.getThreshold();
    }

    /**
     * Returns a new logger instance named as the first parameter using the
     * default factory. If a logger of that name already exists, then it will be
     * returned. Otherwise, a new logger will be instantiated and then linked
     * with its existing ancestors as well as children.
     * 
     * @param clazz the class to get the logger for
     * @return Logger
     */
    public Logger getLogger(Class<?> clazz) {
        return this.hierarchy.getLogger(clazz.getName());
    }

    /**
     * Returns a new logger instance named as the first parameter using the
     * default factory. If a logger of that name already exists, then it will be
     * returned. Otherwise, a new logger will be instantiated and then linked
     * with its existing ancestors as well as children.
     * 
     * @param name logger name
     * @return Logger
     */
    public Logger getLogger(String name) {
        return this.hierarchy.getLogger(name);
    }

    /**
     * The same as getLogger(String name) but using a factory instance instead
     * of a default factory.
     * 
     * @param name logger name
     * @param factory factory instance
     * @return Logger
     */
    public Logger getLogger(String name, LoggerFactory factory) {
        return this.hierarchy.getLogger(name, factory);
    }

    /**
     * Returns the root of this hierarchy.
     * 
     * @return Logger
     */
    public Logger getRootLogger() {
        return this.hierarchy.getRootLogger();
    }

    /**
     * Checks if this logger exists.
     * 
     * @return Logger
     */
    public Logger exists(String name) {
        return this.hierarchy.exists(name);
    }

    /**
     * Disposes the logger hierarchy
     */
    public void shutdown() {
        this.hierarchy.shutdown();
    }

    /**
     * Resets configuration values to its defaults.
     */
    public void resetConfiguration() {
        this.hierarchy.resetConfiguration();
    }

}