package org.simantics.scl.compiler.source.repository;

import java.util.Collection;

import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.PrecompiledModuleSource;

import gnu.trove.map.hash.THashMap;
import gnu.trove.procedure.TObjectProcedure;

/**
 * An implementation of {@link ModuleSourceRepository} as a finite map.
 * This implementation does not support listening module changes,
 * so it should not be modified after it has been taken into use.
 * 
 * @author Hannu Niemist&ouml;
 */
public class MapModuleSourceRepository implements ModuleSourceRepository {
    THashMap<String, ModuleSource> modules = new THashMap<String, ModuleSource>();
    THashMap<String, String> documentations = new THashMap<String, String>();
    
    public MapModuleSourceRepository() {
    }
    
    public MapModuleSourceRepository(ModuleSource ... descriptors) {
        for(ModuleSource descriptor : descriptors)
            addModuleDescriptor(descriptor);
    }
    
    public MapModuleSourceRepository(Module ... modules) {
        for(Module module : modules)
            addModule(module);
    }
    
    public void addModuleDescriptor(ModuleSource descriptor) {
        modules.put(descriptor.getModuleName(), descriptor);
    }
    
    public void addModule(Module module) {
        addModuleDescriptor(new PrecompiledModuleSource(module));
    }
    
    public void addDocumentation(String documentationName, String documentation) {
        documentations.put(documentationName, documentation);
    }

    @Override
    public ModuleSource getModuleSource(String moduleName,
            UpdateListener listener) {
        return modules.get(moduleName);
    }

    @Override
    public Collection<String> getModuleNames() {
        return modules.keySet();
    }
    
    @Override
    public void forAllModules(TObjectProcedure<String> procedure) {
        modules.forEachKey(procedure);
    }

    @Override
    public String getDocumentation(String documentationName) {
        return documentations.get(documentationName);
    }
    
    @Override
    public Collection<String> getDocumentationNames() {
        return documentations.keySet();
    }
}
