package org.simantics.scl.compiler.source.repository;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.source.ModuleSource;

import gnu.trove.procedure.TObjectProcedure;

public class CompositeModuleSourceRepository implements ModuleSourceRepository {
    public final ModuleSourceRepository[] children;

    public CompositeModuleSourceRepository(ModuleSourceRepository ... children) {
        this.children = children;
    }

    @Override
    public ModuleSource getModuleSource(String moduleName,
            UpdateListener listener) {
        ModuleSource result = null;
        for(ModuleSourceRepository sourceRepository : children) {
            ModuleSource source = sourceRepository.getModuleSource(moduleName, listener);
            if(source != null) {
                if(result != null) {
                    double resultPriority = result.getPriority();
                    double sourcePriority = source.getPriority();
                    if(resultPriority > sourcePriority)
                        continue;
                    if(resultPriority == sourcePriority)
                        throw new RuntimeException("Module " + moduleName + " has two sources " + result + ", " +
                                source + " with the same priority.");
                }
                result = source;
            }
        }
        return result;
    }
    
    @Override
    public String getDocumentation(String documentationName) {
        for(ModuleSourceRepository sourceRepository : children) {
            String documentation = sourceRepository.getDocumentation(documentationName);
            if(documentation != null)
                return documentation;
        }
        return null;
    }

    @Override
    public void forAllModules(TObjectProcedure<String> procedure) {
        for(ModuleSourceRepository sourceRepository : children)
            sourceRepository.forAllModules(procedure);
    }
    
    @Override
    public Collection<String> getModuleNames() {
        ArrayList<String> result = new ArrayList<>();
        forAllModules((String name) -> {
            result.add(name);
            return true;
        });
        return result;
    }
    
    @Override
    public void forAllDocumentations(TObjectProcedure<String> procedure) {
        for(ModuleSourceRepository sourceRepository : children)
            sourceRepository.forAllDocumentations(procedure);
    }
    
    @Override
    public Collection<String> getDocumentationNames() {
        ArrayList<String> result = new ArrayList<>();
        forAllDocumentations((String name) -> {
            result.add(name);
            return true;
        });
        return result;
    }

    @Override
    public void checkUpdates() {
        for (ModuleSourceRepository child : children)
            child.checkUpdates();
    }

    @Override
    public void clear() {
        for (ModuleSourceRepository child : children)
            child.clear();
    }
}
