package org.simantics.scl.compiler.environment;

import java.util.function.Consumer;

import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.elaboration.modules.TypeClass;
import org.simantics.scl.compiler.elaboration.modules.TypeDescriptor;
import org.simantics.scl.compiler.elaboration.relations.SCLEntityType;
import org.simantics.scl.compiler.elaboration.relations.SCLRelation;
import org.simantics.scl.compiler.elaboration.rules.MappingRelation;
import org.simantics.scl.compiler.elaboration.rules.TransformationRule;
import org.simantics.scl.compiler.environment.filter.NamespaceFilter;
import org.simantics.scl.compiler.internal.codegen.effects.EffectConstructor;
import org.simantics.scl.compiler.types.TCon;

import gnu.trove.procedure.TObjectProcedure;

public interface Namespace {
	/**
	 * Find a sub-namespace with a given name
	 * @param name  name of the namespace
	 * @return  a Namespace instance, or null if not found
	 */
    Namespace getNamespace(String name);
    
    /**
     * Get an SCLValue for a given name. The same instance is returned on each call.
     * @param name  the name of a defined value
     * @return  The return value is an SCLValue provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    SCLValue getValue(String name) throws AmbiguousNameException;
    
    /**
     * Get an SCLRelation for a given name. The same instance is returned on each call.
     * @param name  the name of a defined relation
     * @return  The return value is an SCLRelation provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    SCLRelation getRelation(String name) throws AmbiguousNameException;
    
    /**
     * Get an SCLEntityType for a given name. The same instance is returned on each call.
     * @param name  the name of a defined entity type
     * @return  The return value is an SCLEntityType provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    SCLEntityType getEntityType(String name) throws AmbiguousNameException;
    
    /**
     * Get a TypeDescriptor for a given name. The same instance is returned on each call.
     * @param name  the name of a defined entity type
     * @return  The return value is a TypeDescriptor provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    TypeDescriptor getTypeDescriptor(String name) throws AmbiguousNameException;
    
    /**
     * Get an EffectConstructor for a given name. The same instance is returned on each call.
     * @param name  an effect name
     * @return  The return value is an EffectConstructor provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    EffectConstructor getEffectConstructor(String name) throws AmbiguousNameException;
    
    /**
     * Get a TypeClass for a given name. The same instance is returned on each call.
     * @param name  the name of a defined entity type
     * @return  The return value is a TypeClass provided by any source included in the namespace,
     *          or null if the name is not found.
     * @exception  AmbiguousNameException  if the name matches with multiple imported modules. 
     */
    TypeClass getTypeClass(String name) throws AmbiguousNameException;
    
    TransformationRule getRule(String name) throws AmbiguousNameException;
    MappingRelation getMappingRelation(String name) throws AmbiguousNameException;

    /**
     * Find all values that share a common prefix and call a procedure for them.
     * @param prefix  A name prefix
     * @param filter  A filter that restricts the set of found values
     * @param proc    A procedure that is called for all found values
     */
    void findValuesForPrefix(String prefix, NamespaceFilter filter, TObjectProcedure<SCLValue> proc);
    
    /**
     * Find all types that share a common prefix and call a consumer for them.
     * @param prefix    A name prefix
     * @param filter    A filter that restricts the set of found values
     * @param consumer  A procedure that is called for all found values
     */
    void findTypesForPrefix(String prefix, NamespaceFilter filter, Consumer<TCon> consumer);
}
