package org.simantics.scl.compiler.elaboration.expressions.list;

import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.expressions.Case;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.expressions.Variable;
import org.simantics.scl.compiler.elaboration.expressions.VariableProcedure;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.types.Type;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class ListAssignment extends ListQualifier {
    public Expression pattern;
    public Expression value;
    
    public ListAssignment(Expression pattern, Expression value) {
        this.pattern = pattern;
        this.value = value;
    }
    
    @Override
    public void checkType(TypingContext context) {
        value = value.inferType(context);
        pattern.checkTypeAsPattern(context, value.getType());
    }

    @Override
    public void collectRefs(TObjectIntHashMap<Object> allRefs,
            TIntHashSet refs) {
        value.collectRefs(allRefs, refs);
    }

    @Override
    public void collectVars(TObjectIntHashMap<Variable> allVars,
            TIntHashSet vars) {
        value.collectVars(allVars, vars);
    }

    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
        value.collectFreeVariables(vars);
        pattern.collectFreeVariables(vars);
    }

    @Override
    public CompiledQualifier compile(SimplificationContext context) {
        if(pattern instanceof EVariable)
            return new CompiledQualifier(context.singletonList(value), pattern);
        else {
            THashSet<Variable> variables = pattern.getFreeVariables();
            Variable[] variableArray = variables.toArray(new Variable[variables.size()]);
            Expression[] variableExps = new Expression[variableArray.length];
            for(int i=0;i<variableArray.length;++i)
                variableExps[i] = new EVariable(variableArray[i]);
            Expression newPattern = context.tuple(variableExps);
            
            EVariable blank = context.blank();
            blank.getVariable().setType(pattern.getType());
            
            return new CompiledQualifier(
                context.match(value, 
                        new Case(pattern, context.singletonList(newPattern.copy())),
                        new Case(blank, context.emptyList(newPattern.getType()))
                        ),
                        newPattern
                    );
        }
    }

    @Override
    public void resolve(TranslationContext context) {
        value = value.resolve(context);
        pattern = pattern.resolveAsPattern(context);
    }

    @Override
    public void decorate(ExpressionDecorator decorator) {
        value = value.decorate(decorator);
        pattern = pattern.decorate(decorator);
    }

    @Override
    public void collectEffects(THashSet<Type> effects) {
        pattern.collectEffects(effects);
        value.collectEffects(effects);
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            pattern.setLocationDeep(loc);
            value.setLocationDeep(loc);
        }
    }

    @Override
    public void accept(ListQualifierVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public void forVariables(VariableProcedure procedure) {
        value.forVariables(procedure);
    }

    @Override
    public ListQualifier accept(ListQualifierTransformer transformer) {
        return transformer.transform(this);
    }
}
