package org.simantics.scl.compiler.elaboration.expressions;

import java.util.ArrayList;

import org.simantics.scl.compiler.common.names.Name;
import org.simantics.scl.compiler.common.precedence.Precedence;
import org.simantics.scl.compiler.constants.Constant;
import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.errors.NotPatternException;
import org.simantics.scl.compiler.elaboration.expressions.lhstype.LhsType;
import org.simantics.scl.compiler.elaboration.expressions.lhstype.PatternMatchingLhs;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.elaboration.utils.ExpressionDecorator;
import org.simantics.scl.compiler.internal.interpreted.IConstant;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.top.SCLCompilerConfiguration;
import org.simantics.scl.compiler.top.ValueNotFound;
import org.simantics.scl.compiler.types.TForAll;
import org.simantics.scl.compiler.types.TMetaVar;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;
import org.simantics.scl.compiler.types.util.MultiFunction;
import org.simantics.scl.compiler.types.util.TypeUnparsingContext;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

public class EConstant extends Expression {
    SCLValue value;
    Type[] typeParameters;
    
    public EConstant(SCLValue value, Type ... typeParameters) {
        if(SCLCompilerConfiguration.DEBUG)
            if(value == null)
                throw new NullPointerException();
        this.value = value;
        this.typeParameters = typeParameters;
    }

    public EConstant(SCLValue value) {
        if(SCLCompilerConfiguration.DEBUG)
            if(value == null)
                throw new NullPointerException();
        this.value = value;
        this.typeParameters = Type.EMPTY_ARRAY;
    }

    public EConstant(long loc, SCLValue value) {
        super(loc);
        if(SCLCompilerConfiguration.DEBUG)
            if(value == null)
                throw new NullPointerException();
        this.value = value;
        this.typeParameters = Type.EMPTY_ARRAY;
    }
    
    public EConstant(long loc, SCLValue value, Type ... typeParameters) {
        super(loc);
        if(SCLCompilerConfiguration.DEBUG)
            if(value == null)
                throw new NullPointerException();
        this.value = value;
        this.typeParameters = typeParameters;
    }

    public void addTypeParameters(Type ... newTypeParameters) {
        typeParameters = Types.concat(typeParameters, newTypeParameters);
    }
    
    public Expression applyType(Type type) {
        typeParameters = Types.concat(typeParameters, new Type[] {type});
        if(getType() != null)
            setType(Types.instantiate(getType(), type));
        return this;
    }

	public void collectRefs(TObjectIntHashMap<Object> allRefs, TIntHashSet refs) {
        int id = allRefs.get(value);
        if(id >= 0)
            refs.add(id);
    }
	
	@Override
	public void collectVars(TObjectIntHashMap<Variable> allVars,
	        TIntHashSet vars) {   
	}

	public void toString(StringBuilder b, TypeUnparsingContext tuc) {
	    Name name = value.getName();
	    if(name.module.equals("Builtin") || name.module.equals("Prelude"))
	        b.append(name.name);
	    else
	        b.append(name);
        /*for(Type type : typeParameters) {
            b.append(" <");
            b.append(type.toString(tuc));
            b.append(">");
        }*/
    }

    @Override
    protected void updateType() throws MatchException {
        setType(Types.instantiate(value.getType(), typeParameters));
    }
    
	@Override
	public IVal toVal(Environment env, CodeWriter w) {
	    IVal val = value.getValue();	        
	    if(typeParameters.length > 0) {
	        val = val.createSpecialization(typeParameters);
	    }
	    return val;
    }

    @Override
    public void collectFreeVariables(THashSet<Variable> vars) {
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        if(value.getInlineInSimplification()) {
            if(typeParameters.length > 0) {
                context.getErrorLog().log(location, 
                        "Inlining with type parameters not currently supported in simplification.");
                return this;
            }
            else
                return value.getExpression().copy().simplify(context);
        }
        return this;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        return this;
    }
    
    @Override
    public void getParameters(TranslationContext translationContext,
            ArrayList<Expression> parameters) {
    }
    
    public SCLValue getValue() {
        return value;
    }
    
    @Override
    public Expression resolveAsPattern(TranslationContext context) {
        return this;
    }
       
    @Override
    public void removeFreeVariables(THashSet<Variable> vars) {     
    }

    @Override
    public Expression replace(ReplaceContext context) {
        Type[] newTypeParameters;
        if(typeParameters.length == 0)
            newTypeParameters = Type.EMPTY_ARRAY;
        else {
            newTypeParameters = new Type[typeParameters.length];
            for(int i=0;i<newTypeParameters.length;++i)
                newTypeParameters[i] = typeParameters[i].replace(context.tvarMap);
        }
        return new EConstant(value, newTypeParameters);
    }
    
    public Type[] getTypeParameters() {
        return typeParameters;
    }

    @Override
    public LhsType getLhsType() throws NotPatternException {
        return new PatternMatchingLhs();
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext target) {
        Name name = value.getName();
        try {
            return new IConstant(target.runtimeEnvironment.getRuntimeModule(name.module).getValue(name.name));
        } catch (ValueNotFound e) {
            throw new UnsupportedOperationException();
        }
    }
    
    @Override
    public Expression inferType(TypingContext context) {
        if(context.recursiveValues != null &&
                context.recursiveValues.contains(value)) {
            // Handles the case where the constant is one of the recursive definitions we are currently checking
            // This kind of value is not yet generalized, i.e. it is not necessary to instantiate it.
            EPlaceholder placeholder = new EPlaceholder(location, this);
            placeholder.setType(value.getType());
            
            context.recursiveReferences.add(placeholder);
            return placeholder;
        }
        else if(context.isInPattern()) {
            /* This is little hackish code that handles the following kind of constructors:
             *   data Thunk a = Thunk s (a -> s)
             * in
             *   match thunk with Thunk s f -> f s
             * We cannot assign s with an unbound metaVar because its type depends on 
             * how it has been constructed. Therefore we parametrize the function with
             * existential variable.
             */
            Type resultType = value.getType();
            if(resultType instanceof TForAll) {
                ArrayList<TMetaVar> vars = new ArrayList<TMetaVar>(); 
                resultType = Types.instantiate(resultType, vars);
                MultiFunction mfun = Types.matchFunction(resultType);
                resultType = mfun.returnType;
                
                for(TMetaVar var : vars) {
                    if(resultType.contains(var))
                        break;
                    addTypeParameters(Types.var(var.getKind()));                    
                }
            }
            return this;
        }
        else
            return applyPUnit(context);
    }

    @Override
    public Expression decorate(ExpressionDecorator decorator) {
        return decorator.decorate(this);
    }
    
    @Override
    public boolean isEffectful() {
    	return false;
    }

    @Override
    public void collectEffects(THashSet<Type> effects) {
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION)
            location = loc;
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public Precedence getPrecedence() {
        return value.getPrecedence();
    }

    @Override
    public void forVariables(VariableProcedure procedure) {
    }
    
    @Override
    public boolean isPattern(int arity) {
        IVal val = value.getValue();
        if(!(val instanceof Constant))
            return false;
        Constant constant = (Constant)val;
        return constant.constructorTag() >= 0 && constant.getArity() == arity;
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }

    @Override
    public boolean equalsExpression(Expression expression) {
        if(expression.getClass() != getClass())
            return false;
        EConstant other = (EConstant)expression;
        return value == other.value && Types.equals(typeParameters, other.typeParameters);
    }
}
