package org.simantics.scl.compiler.elaboration.chr.planning;

import java.util.ArrayList;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.chr.CHRLiteral;
import org.simantics.scl.compiler.elaboration.chr.plan.AccessFactOp;
import org.simantics.scl.compiler.elaboration.chr.plan.ClaimOp;
import org.simantics.scl.compiler.elaboration.chr.plan.ExecuteOp;
import org.simantics.scl.compiler.elaboration.chr.plan.PlanOp;
import org.simantics.scl.compiler.elaboration.chr.planning.items.CheckPrePlanItem;
import org.simantics.scl.compiler.elaboration.chr.planning.items.EqualsPrePlanItem;
import org.simantics.scl.compiler.elaboration.chr.planning.items.GenericPrePlanItem;
import org.simantics.scl.compiler.elaboration.chr.planning.items.MemberPrePlanItem;
import org.simantics.scl.compiler.elaboration.chr.relations.CHRConstraint;
import org.simantics.scl.compiler.elaboration.chr.relations.SpecialCHRRelation;
import org.simantics.scl.compiler.elaboration.expressions.EApplyType;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.EExternalConstant;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.expressions.Variable;

import gnu.trove.impl.Constants;
import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

public class QueryPlanningContext {
    CompilationContext compilationContext;
    public PlanPriorityQueue priorityQueue = new PlanPriorityQueue();
    ArrayList<Variable> variables;
    TObjectIntHashMap<Variable> variableMap;
    ArrayList<ArrayList<PrePlanItem>> itemsContainingVariable;
    ArrayList<PlanOp> planOps = new ArrayList<PlanOp>(); 
    
    public QueryPlanningContext(CompilationContext compilationContext, Variable[] variables) {
        this.compilationContext = compilationContext;
        this.variables = new ArrayList<Variable>(variables.length*2);
        this.variableMap = new TObjectIntHashMap<Variable>(variables.length, Constants.DEFAULT_LOAD_FACTOR, -1);
        itemsContainingVariable = new ArrayList<ArrayList<PrePlanItem>>(variables.length*2);
        for(Variable variable : variables)
            addVariable(variable);
    }
    
    private void addVariable(Variable variable) {
        int id = variables.size();
        variables.add(variable);
        variableMap.put(variable, id);
        itemsContainingVariable.add(new ArrayList<PrePlanItem>(2));
    }

    public void add(CHRLiteral literal, int secondaryPriority) {
        if(literal.relation instanceof SpecialCHRRelation) {
            switch((SpecialCHRRelation)literal.relation) {
            case CHECK:
                addCheck(literal.location, literal.parameters[0], secondaryPriority);
                return;
            case EQUALS:
                addGenericEquals(literal.location, literal.parameters[0], literal.parameters[1], secondaryPriority);
                return;
            case MEMBER:
                addMember(literal.location, literal.parameters[0], literal.parameters[1], secondaryPriority);
                return;
            case EXECUTE:
                throw new InternalCompilerError(literal.location, "EXECUTE constraint is not allowed in query compilation.");
            }
        }
        
        addGenericConstraint(literal, secondaryPriority);
    }
    
    private TIntHashSet getVars(Expression expression, int initialCapacity) {
        TIntHashSet variableSet = new TIntHashSet(initialCapacity);
        expression.collectVars(variableMap, variableSet);
        return variableSet;
    }
    
    private TIntHashSet[] getVars(Expression[] expressions, int initialCapacity) {
        TIntHashSet[] variableSets = new TIntHashSet[expressions.length];
        for(int i=0;i<expressions.length;++i)
            variableSets[i] = getVars(expressions[i], initialCapacity);
        return variableSets;
    }
    
    /**
     * Returns true, if the expression is so simple, it does not involve any computation.
     */
    private static boolean isSimpleExpression(Expression expression) {
        while(expression instanceof EApplyType)
            expression = ((EApplyType)expression).getExpression();
        return expression instanceof EVariable 
                || expression instanceof EConstant
                || expression instanceof ELiteral
                || expression instanceof EExternalConstant;
    }
    
    private Expression toSimpleExpression(Expression expression, int secondaryPriority) {
        if(isSimpleExpression(expression))        
            return expression;
        else {
            Variable temp = new Variable("temp", expression.getType());
            addVariable(temp);
            addOneSidedEquals(expression.location, new EVariable(temp), expression, secondaryPriority);
            return new EVariable(temp);
        }
    }
    
    private Expression[] toSimpleExpressions(Expression[] expressions, int secondaryPriority) {
        Expression[] result = new Expression[expressions.length];
        for(int i=0;i<expressions.length;++i)
            result[i] = toSimpleExpression(expressions[i], secondaryPriority);
        return result;
    }

    private void addGenericConstraint(CHRLiteral literal, int secondaryPriority) {
        if(literal.killAfterMatch)
            ((CHRConstraint)literal.relation).setMayBeRemoved();
        Expression[] parameters = toSimpleExpressions(literal.parameters, secondaryPriority);
        add(literal.location, new GenericPrePlanItem(literal, literal.relation, parameters, getVars(parameters, 1), secondaryPriority));
    }

    private void addMember(long location, Expression p1, Expression p2, int secondaryPriority) {
        Expression expression1 = toSimpleExpression(p1, secondaryPriority);
        Expression expression2 = toSimpleExpression(p2, secondaryPriority);
        add(location, new MemberPrePlanItem(expression1, expression2,
                getVars(expression1, 1), getVars(expression2, 1), secondaryPriority));
    }
    
    private void addOneSidedEquals(long location, Expression expression1, Expression expression2, int secondaryPriority) {
        add(location, new EqualsPrePlanItem(expression1, expression2,
                getVars(expression1, 1), getVars(expression2, 4), secondaryPriority));
    }

    private void addGenericEquals(long location, Expression p1, Expression p2, int secondaryPriority) {
        if(isSimpleExpression(p1))
            addOneSidedEquals(location, p1, p2, secondaryPriority);
        else if(isSimpleExpression(p2))
            addOneSidedEquals(location, p2, p1, secondaryPriority);
        else {
            Variable temp = new Variable("temp", p1.getType());
            addVariable(temp);
            addOneSidedEquals(p1.location, new EVariable(temp), p1, secondaryPriority);
            addOneSidedEquals(p2.location, new EVariable(temp), p2, secondaryPriority);
        }
    }

    private void addCheck(long location, Expression condition, int secondaryPriority) {
        TIntHashSet variableSet = new TIntHashSet(4);
        condition.collectVars(variableMap, variableSet);
        add(location, new CheckPrePlanItem(condition, variableSet, secondaryPriority));
    }

    private void add(long location, PrePlanItem item) {
        priorityQueue.add(item);
        item.initializeListeners(this);
        item.location = location;
    }

    public void listen(TIntHashSet variableSet, PrePlanItem item) {
        variableSet.forEach(new TIntProcedure() {
            @Override
            public boolean execute(int variableId) {
                listen(variableId, item);
                return true;
            }
        });
    }

    public void listen(int variableId, PrePlanItem item) {
        itemsContainingVariable.get(variableId).add(item);
    }

    public boolean createQueryPlan() {
        while(!priorityQueue.isEmpty()) {
            PrePlanItem head = priorityQueue.head();
            if(head.primaryPriority == Double.POSITIVE_INFINITY) {
                compilationContext.errorLog.log(head.location, "Cannot solve the query.");
                return false;
            }
            priorityQueue.pop();
            head.generate(this);
        }
        return true;
    }
    
    public ArrayList<PlanOp> getPlanOps() {
        return planOps;
    }
    
    private final TIntProcedure BIND_PROCEDURE = new TIntProcedure() {
        @Override
        public boolean execute(int variableId) {
            ArrayList<PrePlanItem> l = itemsContainingVariable.get(variableId);
            for(PrePlanItem item : l)
                item.variableSolved(QueryPlanningContext.this, variableId);
            l.clear();
            return true;
        }
    };

    public void bind(TIntHashSet variableSet) {
        variableSet.forEach(BIND_PROCEDURE);
    }

    public void addPlanOp(PlanOp planOp) {
        planOps.add(planOp);
    }

    public CompilationContext getCompilationContext() {
        return compilationContext;
    }

    public void activate(CHRLiteral literal, Expression inputFact, int secondaryPriority) {
        Variable[] variables = new Variable[literal.parameters.length];
        for(int i=0;i<literal.parameters.length;++i)
            variables[i] = new Variable("activeFactComponent" + i, literal.parameters[i].getType());
        if(literal.killAfterMatch)
            ((CHRConstraint)literal.relation).setMayBeRemoved();
        planOps.add(new AccessFactOp(literal.location, inputFact, (CHRConstraint)literal.relation, variables, literal.killAfterMatch));
        for(int i=0;i<literal.parameters.length;++i)
            addOneSidedEquals(literal.parameters[i].location, new EVariable(variables[i]), literal.parameters[i], secondaryPriority);
    }

    public void claim(QueryPlanningContext context, CHRLiteral literal) {
        if(literal.relation instanceof CHRConstraint) {
            CHRConstraint constraint = (CHRConstraint)literal.relation;
            addPlanOp(new ClaimOp(literal.location, constraint, literal.parameters));
        }
        else if(literal.relation instanceof SpecialCHRRelation) {
            switch((SpecialCHRRelation)literal.relation) {
            case EXECUTE:
                addPlanOp(new ExecuteOp(literal.location, literal.parameters[0]));
                break;
            default:
                context.getCompilationContext().errorLog.log(
                        literal.location,
                        "Cannot enforce this constraint.");
            }
        }
    }

}
